/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui;

import com.google.common.eventbus.Subscribe;
import java.io.IOException;
import java.io.StringReader;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.regex.Pattern;
import javafx.print.PrinterJob;
import javafx.scene.Node;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.MenuItem;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.SeparatorMenuItem;
import javafx.scene.input.ClipboardContent;
import javafx.scene.input.Dragboard;
import javafx.scene.input.KeyEvent;
import javafx.scene.input.TransferMode;
import javafx.scene.web.WebView;
import org.jabref.Globals;
import org.jabref.gui.BasePanel;
import org.jabref.gui.ClipBoardManager;
import org.jabref.gui.DialogService;
import org.jabref.gui.EntryContainer;
import org.jabref.gui.FXDialogService;
import org.jabref.gui.IconTheme;
import org.jabref.gui.keyboard.KeyBinding;
import org.jabref.gui.keyboard.KeyBindingRepository;
import org.jabref.gui.util.BackgroundTask;
import org.jabref.gui.util.DefaultTaskExecutor;
import org.jabref.logic.citationstyle.CitationStyle;
import org.jabref.logic.exporter.ExporterFactory;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.layout.Layout;
import org.jabref.logic.layout.LayoutHelper;
import org.jabref.logic.search.SearchQueryHighlightListener;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.event.FieldChangedEvent;
import org.jabref.preferences.PreviewPreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PreviewPanel
extends ScrollPane
implements SearchQueryHighlightListener,
EntryContainer {
    private static final Logger LOGGER = LoggerFactory.getLogger(PreviewPanel.class);
    private final ClipBoardManager clipBoardManager;
    private final DialogService dialogService;
    private final KeyBindingRepository keyBindingRepository;
    private Optional<BasePanel> basePanel = Optional.empty();
    private boolean fixedLayout;
    private Optional<Layout> layout = Optional.empty();
    private Optional<BibEntry> bibEntry = Optional.empty();
    private Optional<BibDatabaseContext> databaseContext = Optional.empty();
    private WebView previewView;
    private Optional<Future<?>> citationStyleFuture = Optional.empty();

    public PreviewPanel(BasePanel panel, BibDatabaseContext databaseContext) {
        this.databaseContext = Optional.ofNullable(databaseContext);
        this.basePanel = Optional.ofNullable(panel);
        this.clipBoardManager = new ClipBoardManager();
        this.dialogService = new FXDialogService();
        this.keyBindingRepository = Globals.getKeyPrefs();
        DefaultTaskExecutor.runInJavaFXThread(() -> {
            this.setFitToHeight(true);
            this.setFitToWidth(true);
            this.previewView = new WebView();
            this.setContent((Node)this.previewView);
            this.previewView.setContextMenuEnabled(false);
            this.setContextMenu(this.createPopupMenu());
            if (this.basePanel.isPresent()) {
                this.setOnDragDetected(event -> {
                    Dragboard dragboard = this.startDragAndDrop(new TransferMode[]{TransferMode.COPY});
                    ClipboardContent content = new ClipboardContent();
                    content.putHtml((String)this.previewView.getEngine().executeScript("window.getSelection().toString()"));
                    dragboard.setContent((Map)content);
                    event.consume();
                });
            }
            this.createKeyBindings();
            this.updateLayout();
        });
    }

    private void createKeyBindings() {
        this.addEventFilter(KeyEvent.KEY_PRESSED, event -> {
            Optional<KeyBinding> keyBinding = Globals.getKeyPrefs().mapToKeyBinding((KeyEvent)event);
            if (keyBinding.isPresent()) {
                switch (keyBinding.get()) {
                    case COPY_PREVIEW: {
                        this.copyPreviewToClipBoard();
                        event.consume();
                        break;
                    }
                    case CLOSE_DIALOG: {
                        this.close();
                        event.consume();
                        break;
                    }
                }
            }
        });
    }

    private ContextMenu createPopupMenu() {
        MenuItem copyPreview = new MenuItem(Localization.lang("Copy preview", new String[0]), IconTheme.JabRefIcon.COPY.getGraphicNode());
        copyPreview.setAccelerator(this.keyBindingRepository.getKeyCombination(KeyBinding.COPY_PREVIEW));
        copyPreview.setOnAction(event -> this.copyPreviewToClipBoard());
        MenuItem printEntryPreview = new MenuItem(Localization.lang("Print entry preview", new String[0]), IconTheme.JabRefIcon.PRINTED.getGraphicNode());
        printEntryPreview.setOnAction(event -> this.print());
        MenuItem previousPreviewLayout = new MenuItem(Localization.menuTitleFX("Previous preview layout", new String[0]));
        previousPreviewLayout.setAccelerator(this.keyBindingRepository.getKeyCombination(KeyBinding.PREVIOUS_PREVIEW_LAYOUT));
        previousPreviewLayout.setOnAction(event -> this.basePanel.ifPresent(BasePanel::previousPreviewStyle));
        MenuItem nextPreviewLayout = new MenuItem(Localization.menuTitleFX("Next preview layout", new String[0]));
        nextPreviewLayout.setAccelerator(this.keyBindingRepository.getKeyCombination(KeyBinding.NEXT_PREVIEW_LAYOUT));
        nextPreviewLayout.setOnAction(event -> this.basePanel.ifPresent(BasePanel::nextPreviewStyle));
        ContextMenu menu = new ContextMenu();
        menu.getItems().add((Object)copyPreview);
        menu.getItems().add((Object)printEntryPreview);
        menu.getItems().add((Object)new SeparatorMenuItem());
        menu.getItems().add((Object)nextPreviewLayout);
        menu.getItems().add((Object)previousPreviewLayout);
        return menu;
    }

    public void setDatabaseContext(BibDatabaseContext databaseContext) {
        this.databaseContext = Optional.ofNullable(databaseContext);
    }

    public Optional<BasePanel> getBasePanel() {
        return this.basePanel;
    }

    public void setBasePanel(BasePanel basePanel) {
        this.basePanel = Optional.ofNullable(basePanel);
    }

    public void updateLayout(PreviewPreferences previewPreferences) {
        if (this.fixedLayout) {
            LOGGER.debug("cannot change the layout because the layout is fixed");
            return;
        }
        String style = previewPreferences.getPreviewCycle().get(previewPreferences.getPreviewCyclePosition());
        if (CitationStyle.isCitationStyleFile(style)) {
            if (this.basePanel.isPresent()) {
                this.layout = Optional.empty();
                CitationStyle.createCitationStyleFromFile(style).ifPresent(citationStyle -> {
                    this.basePanel.get().getCitationStyleCache().setCitationStyle((CitationStyle)citationStyle);
                    this.basePanel.get().output(Localization.lang("Preview style changed to: %0", citationStyle.getTitle()));
                });
            }
        } else {
            this.updatePreviewLayout(previewPreferences.getPreviewStyle());
            this.basePanel.ifPresent(panel -> panel.output(Localization.lang("Preview style changed to: %0", Localization.lang("Preview", new String[0]))));
        }
        this.update();
    }

    public void updateLayout() {
        this.updateLayout(Globals.prefs.getPreviewPreferences());
    }

    private void updatePreviewLayout(String layoutFile) {
        StringReader sr = new StringReader(layoutFile.replace("__NEWLINE__", "\n"));
        try {
            this.layout = Optional.of(new LayoutHelper(sr, Globals.prefs.getLayoutFormatterPreferences(Globals.journalAbbreviationLoader)).getLayoutFromText());
        }
        catch (IOException e) {
            this.layout = Optional.empty();
            LOGGER.debug("no layout could be set", e);
        }
    }

    public void setLayout(Layout layout) {
        this.layout = Optional.ofNullable(layout);
        this.update();
    }

    public void setEntry(BibEntry newEntry) {
        this.bibEntry.filter(e -> e != newEntry).ifPresent(e -> e.unregisterListener(this));
        this.bibEntry = Optional.ofNullable(newEntry);
        newEntry.registerListener(this);
        this.update();
    }

    @Subscribe
    public void listen(FieldChangedEvent fieldChangedEvent) {
        this.update();
    }

    @Override
    public BibEntry getEntry() {
        return this.bibEntry.orElse(null);
    }

    public void update() {
        ExporterFactory.entryNumber = 1;
        if (this.citationStyleFuture.isPresent()) {
            this.citationStyleFuture.get().cancel(true);
            this.citationStyleFuture = Optional.empty();
        }
        if (this.layout.isPresent()) {
            StringBuilder sb = new StringBuilder();
            this.bibEntry.ifPresent(entry -> sb.append(this.layout.get().doLayout((BibEntry)entry, this.databaseContext.map(BibDatabaseContext::getDatabase).orElse(null))));
            this.setPreviewLabel(sb.toString());
        } else if (this.basePanel.isPresent() && this.bibEntry.isPresent()) {
            Future<?> citationStyleWorker = BackgroundTask.wrap(() -> this.basePanel.get().getCitationStyleCache().getCitationFor(this.bibEntry.get())).onRunning(() -> {
                CitationStyle citationStyle = this.basePanel.get().getCitationStyleCache().getCitationStyle();
                this.setPreviewLabel("<i>" + Localization.lang("Processing %0", Localization.lang("Citation Style", new String[0])) + ": " + citationStyle.getTitle() + " ...</i>");
            }).onSuccess(this::setPreviewLabel).onFailure(exception -> {
                LOGGER.error("Error while generating citation style", (Throwable)exception);
                this.setPreviewLabel(Localization.lang("Error while generating citation style", new String[0]));
            }).executeWith(Globals.TASK_EXECUTOR);
            this.citationStyleFuture = Optional.of(citationStyleWorker);
        }
    }

    private void setPreviewLabel(String text) {
        DefaultTaskExecutor.runInJavaFXThread(() -> {
            this.previewView.getEngine().loadContent(text);
            this.setHvalue(0.0);
        });
    }

    @Override
    public void highlightPattern(Optional<Pattern> newPattern) {
        this.update();
    }

    public void setFixedLayout(String layout) {
        this.fixedLayout = true;
        this.updatePreviewLayout(layout);
    }

    public void print() {
        PrinterJob job = PrinterJob.createPrinterJob();
        boolean proceed = this.dialogService.showPrintDialog(job);
        if (!proceed) {
            return;
        }
        BackgroundTask.wrap(() -> {
            job.getJobSettings().setJobName(this.bibEntry.flatMap(BibEntry::getCiteKeyOptional).orElse("NO ENTRY"));
            this.previewView.getEngine().print(job);
            job.endJob();
            return null;
        }).onFailure(exception -> this.dialogService.showErrorDialogAndWait(Localization.lang("Could not print preview", new String[0]), (Throwable)exception)).executeWith(Globals.TASK_EXECUTOR);
    }

    public void close() {
        this.basePanel.ifPresent(BasePanel::hideBottomComponent);
    }

    private void copyPreviewToClipBoard() {
        String previewContent = (String)this.previewView.getEngine().executeScript("document.documentElement.outerHTML");
        this.clipBoardManager.setClipboardContents(previewContent);
    }
}

