(* (c) Microsoft Corporation 2005-2006.  *)

#light
#nowarn "0049";; // turn off warnings about using upper case identifiers for variables (e.g. matrices)

#load "library.fs";;

open Microsoft.FSharp.Math.Bindings.LAPACK
open Microsoft.FSharp.Math.Notation
open Microsoft.FSharp.Math

// Ensure you have LAPACK.dll and BLAS.dll in the sample directory or elsewhere
// on your path. Here we set the current directory so we can find any local copies
// of LAPACK.dll and BLAS.dll.
System.Environment.CurrentDirectory <- __SOURCE_DIRECTORY__ 

// Here are some simple matrix values.
let onesM = matrix [ [ 1.0; 1.0];  
                      [1.0; 1.0] ]
let onesA = [| 1.0; 1.0 |]
let onesV = vector [ 1.0; 1.0]
let onesv = matrix [ [ 1.0] ; 
                     [ 1.0] ]
let twosM = matrix [ [ 2.0; 2.0]; 
                     [ 2.0; 2.0] ]
let twosV = vector [ 2.0; 2.0 ]
let twosv = matrix [ [ 2.0] ; 
                     [ 2.0] ]
let iM = matrix [ [ 1.0; 0.0]; 
                  [ 0.0; 1.0] ]

let miM = matrix [ [ -1.0; 0.0]; 
                   [  0.0; -1.0] ]

matrix [ [ 0.0; 0.0]; 
         [ 0.0; 0.0] ]

Matrix.identity 2

Matrix.identity 10

let A = 1

let show x = printf "%s\n" (any_to_string x)
printf " ------------------------\n" 

let J  = matrix [ [ 2.0; 3.0]; 
                  [ 4.0; 5.0] ]
let J2 = matrix [ [ 2.0; 3.0;4.0]; 
                  [ 2.0; 3.0;5.0]; 
                  [ 4.0; 5.0;6.0] ]

// MATALB notation is M \ v, i.e. solve Mx=v for vector x.  Solving Mx=v
// The notation we use here is M $ v
let ( $ ) (M : matrix) v = ImmutableMatrixRoutines.solve M v
let ( *. ) A B = ImmutableMatrixRoutines.mulMM A B
let ( *%. ) A v = ImmutableMatrixRoutines.mulMV A v

let Random = Matrix.create 500 500 1.0 |> Matrix.randomize
let RandomV = Matrix.create 500 1 1.0 |> Matrix.to_vector

let time s f =
  let sw = new System.Diagnostics.Stopwatch() 
  sw.Start();
  let res = f()
  printf "%A, time: %d\n" res sw.ElapsedMilliseconds;
  res
  
  
time "500x500 Matrix-MAtrix Multiplication using F# built-in: Random * Random" (fun () -> Random * Random )
time "500x500 Matrix-Matrix Multiplication using LAPACK.dll: Random *. Random" (fun () -> Random *. Random )
time "500x500 Matrix-Vector Multiplication using LAPACK.dll:  Random $ Random" (fun () -> Random $ RandomV )
time "Checking results correlate, err = " (fun () -> Vector.sum (Random * (Random $ RandomV) - RandomV))
time "Solving system of 500 equations in 500 variables, check with *%, err = " (fun () -> Vector.sum (Random *%. (Random $ RandomV) - RandomV))

time "computeEigenValues Random" (fun () -> ImmutableMatrixRoutines.computeEigenValues Random)

(*
ImmutableMatrixRoutines.computeEigenValues (Matrix.identity 2)
ImmutableMatrixRoutines.computeEigenValuesAndVectors (Matrix.identity 2)
ImmutableMatrixRoutines.computeEigenValuesAndVectors (matrix [ [ 3.0 ] ])
ImmutableMatrixRoutines.computeEigenValuesAndVectors (matrix [ [ 3.0; 0.0 ]; [ 0.0; 3.0] ])
ImmutableMatrixRoutines.computeEigenValuesAndVectors (matrix [ [ 1.0; 0.0 ]; [ 0.0; -1.0] ])
ImmutableMatrixRoutines.computeEigenValuesAndVectors (matrix [ [ -1.0; 0.0 ]; [ 0.0; -1.0] ])

open System.Windows.Forms
open System.Drawing
let f = new Form()
f.Visible <- true
f.TopMost <- true
let mutable trans = fun (t:float) -> Matrix.identity 2
let mutable points = [ (0.0, 0.0); (1.0,0.0); (1.0,1.0); (0.0,1.0); (0.0,0.0) ]

f.Paint.Add(fun e -> 
    let t = (System.DateTime.Now.Ticks |> float) / 10000000.0
    let trans = trans t
    let of_vec (v:vector) = new Point(f.Width/2 + truncate (v.[0]*100.0), f.Height/2  - truncate (v.[1]*100.0)) 
    let origin = vector [0.0; 0.0]
    let to_vec (x,y) = vector [x;y]
    let draw_vec pen v1 v2 = e.Graphics.DrawLine(pen,of_vec v1,of_vec v2);
    let draw (p1,p2) = draw_vec Pens.Aqua (trans * to_vec p1) (trans * to_vec p2);
    let es,E = ImmutableMatrixRoutines.computeEigenValuesAndVectors trans.Transpose
    draw_vec Pens.Red origin (Matrix.getCol E 0 * es.[0].r)
    draw_vec Pens.Red origin (Matrix.getCol E 1 * es.[1].r)
    points |> List.fold_left (fun acc p1 -> match acc with None -> Some p1 | Some p0 -> draw (p0,p1); Some p1) None |> ignore;
    f.Invalidate())
let scaleM n k = Matrix.diag (Vector.create n k)
trans <- (fun t -> scaleM 2 (sin t)); f.Invalidate()
let rotateM th = matrix [ [ cos th; -sin th ]; [ sin th; cos th ] ]
let pi = System.Math.PI
trans <- (fun t -> rotateM (pi * t)); f.Invalidate()
trans <- (fun t -> matrix [ [ sin t; 1.0]; [cos (t/1.3); 1.0] ])

ImmutableMatrixRoutines.computeEigenValuesAndVectors (rotateM 0.23)
let M = matrix [ [ 0.5; 1.0]; [0.0; 1.0] ]
trans <- (fun _ -> M)
let es,EM = ImmutableMatrixRoutines.computeEigenValuesAndVectors M.Transpose

let e0 = es.[1]
assert(e0.i = 0.0) 
M.Transpose * Matrix.getCol EM 0
M.Transpose * Matrix.getCol EM 1 *$ (1.0/es.[1].r)
let disp x = any_to_string x
*)

(*

printf " ------------------------\n" 
printf " %s $ %s = %s\n" (disp J) (disp onesV) (disp (J $ onesV))
printf " ------------------------\n" 
onesV  |> show
printf " ------------------------\n" 
J * (J $ vector [1.0;1.0])  |> show
printf " ------------------------\n" 
J2 * (J2 $ vector [1.0;2.0;3.0])  |> show
printf " ------------------------\n" 
printf " DONE\n" 
*)

System.Windows.Forms.MessageBox.Show("Done!") |> ignore

(*
#if COMPILED
[<System.STAThread()>]
do Application.Run(f)
#endif
*)
