// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 

// FSharp sample - text editor 

#light

open System
open System.IO  
open System.Windows.Forms

let form = new Form()
form.Width  <- 400
form.Height <- 300
form.Visible <- true 
form.Text <- "F# Editor Sample"

// menu bar and menus 
let mMain = form.Menu <- new MainMenu()
let mFile = form.Menu.MenuItems.Add("&File")
let mHelp = form.Menu.MenuItems.Add("&Help")

// menu items 
let miOpen  = new MenuItem("&Open...")
let miSave  = new MenuItem("&Save As...")
let miQuit  = new MenuItem("&Quit")
let miAbout = new MenuItem("&About...")

mFile.MenuItems.Add(miOpen)
mFile.MenuItems.Add(miSave)
mFile.MenuItems.Add(miQuit)
mHelp.MenuItems.Add(miAbout)

// RichTextBox - text area 
let textB = new RichTextBox()
textB.Dock <- DockStyle.Fill
form.Controls.Add(textB)

// text state 
let setText s = textB.Text <- s
let getText s = textB.Text

// filename state 
let filename = ref ""
let setFilename f  = filename := f; form.Text <- "Editor - " ^ f
let getFilename () = !filename

// initial state   
setFilename "scratch.txt"
setText     "type text in here..."

// readFile dialog 
let readFile () =
    let d = new OpenFileDialog() in 
    d.Filter <- "text files *.txt|*.txt|All files *.*|*.*";
    d.FilterIndex <- 2;
    if d.ShowDialog() = DialogResult.OK then
        let str  = new StreamReader(d.FileName) in
        let text = str.ReadToEnd () in
        Some (d.FileName,text)
    else
        None

// saveFile dialog 
let saveFile file text =
    let d = new SaveFileDialog() in 
    d.Filter <- "text files *.txt|*.txt|All files *.*|*.*";
    d.FilterIndex <- 2;
    d.FileName <- file;
    d.AddExtension <- false;  
    d.OverwritePrompt <- true;
    if d.ShowDialog() = DialogResult.OK then
        let sw = new StreamWriter(d.FileName) in
        sw.Write(text:string); // type constrain text to resolve sw.Write(_) method overload 
        sw.Close();
        Some d.FileName
    else
        None

// ops 
let opLoadText _ = 
    match readFile () with
    | Some (file,text) -> setFilename file; setText text
    | None -> ()

let opSaveText _ = 
    match saveFile (getFilename()) (getText()) with
    | Some file -> setFilename file
    | None -> ()

let opAbout _ = 
    MessageBox.Show("Hello Winforms F# Sample!","About Sample") |> ignore

let opExitForm _ = form.Close ()

// callbacks 
let _ = miOpen.Click.Add(opLoadText)
let _ = miSave.Click.Add(opSaveText)
let _ = miQuit.Click.Add(opExitForm)
let x = miAbout.Click.Add(opAbout)

#if COMPILED
// run!   
// Run the main code. The attribute marks the startup application thread as "Single 
// Thread Apartment" mode, which is recommended for .NET GUI applications. 
[<STAThread()>]    
do Application.Run(form)
#endif

