(* (c) Microsoft Corporation. All rights reserved *)

(*F#
/// Blobs of bytes, cross-compiling 
module Microsoft.Research.AbstractIL.Internal.Bytes 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)

(*F# type bytes = byte[] F#*)
(*IF-OCAML*) type bytes  (*ENDIF-OCAML*)

val length: bytes -> int
val get: bytes -> int -> int    (* returned int will be 0 <= x <= 255 *)
val make: (int -> int) -> int -> bytes
val zero_create: int -> bytes
val of_intarray: int array ->  bytes (* each int must be 0 <= x <= 255 *)
val to_intarray: bytes -> int array (* each int will be 0 <= x <= 255 *)
val sub: bytes -> int -> int -> bytes
val set: bytes -> int -> int -> unit
val blit: bytes -> int -> bytes -> int -> int -> unit
val append: bytes -> bytes -> bytes
val compare: bytes -> bytes -> int


(* Read/write bytes off a binary stream *)
val really_input: in_channel -> int -> bytes
val maybe_input: in_channel -> int -> bytes
val output: out_channel -> bytes -> unit

(* Bytes are commonly used for unicode strings *)
val string_as_unicode_bytes: string -> bytes
val string_as_utf8_bytes: string -> bytes
val unicode_bytes_as_string: bytes -> string
val utf8_bytes_as_string: bytes -> string

(* included mainly for legacy reasons *)
val string_as_unicode_bytes_null_terminated: string -> bytes
val string_as_utf8_bytes_null_terminated: string -> bytes


(* Imperative buffers and streams of bytes *)
module Bytebuf :sig

    type t 
    val create : int -> t
    val emit_int_as_byte : t -> int -> unit
    val emit_intarray_as_bytes : t -> int array -> unit
    val close : t -> bytes
    val emit_bytes : t -> bytes -> unit
    val emit_i32 : t -> int32 -> unit
    val emit_i32_as_u16 : t -> int32 -> unit
    val length : t -> int
end


module Bytestream :sig
    type t
    val of_bytes: bytes -> int -> int -> t
    val read_byte: t -> int
    val read_bytes: t -> int -> bytes
    val read_unicode_bytes_as_string: t -> int -> string
    val read_utf8_bytes_as_string: t -> int -> string
end
