(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
/// ILX extensions to Abstract IL types and instructions F# 
module Microsoft.Research.AbstractIL.Extensions.ILX.Types
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  

open Il
open Nums

(*F#
/// ILX extensions to the algebra of types
///
/// The extended array type is only relevant for ILX-style generics.  
/// We erase ty[] to object[] for all array types
/// built with this array constructor.  If 
/// false, it is the normal .NET co-variant array 
/// constructor for the given bounds.  Languages with generics 
/// over arrays should compile their array types with 
/// the flag set to "true".  Note that when the flag is true types such 
/// as "int array" or "string array" will be compiled 
/// to "object[]" and will NOT be compatible with 
/// the corresponding .NET array types.  This effectively places a 
/// language-level restriction on interop when erasing generics, but 
/// this is to be expected when erasing: you only get real polymorphism 
/// types when you have real generics in the type system of the CLR, 
/// i.e. a generic CLR  
F#*)
type ilx_typ =
 | EType_erasable_array of array_shape * typ      (* -- Pseudo-array types *)

val mk_ilx_ext_typ: (ilx_typ -> ext_typ)
val is_ilx_ext_typ: (ext_typ -> bool)
val dest_ilx_ext_typ: (ext_typ -> ilx_typ)

val mk_ilx_typ: ilx_typ -> typ

(* -------------------------------------------------------------------- 
 * Class union and closure references 
 * -------------------------------------------------------------------- *)

type alternative = 
    { altName: string;
      altFields: field_def array;
      altCustomAttrs: custom_attrs }

and classunion_ref = ClassunionRef of type_ref * alternative array * bool (* cudNullPermitted *)
and classunion_spec = ClassunionSpec of classunion_ref * genactuals

and lambdas = 
  | Lambdas_forall of genparam * lambdas
  | Lambdas_lambda of param * lambdas
  | Lambdas_return of typ
and closure_ref = ClosureRef of type_ref * lambdas * freevar list 
and closure_spec = ClosureSpec of closure_ref * genactuals
and freevar = 
    { fvName: string ; fvType: typ }


(* -------------------------------------------------------------------- 
 * apps - i.e. types being applied at a callsite
 * -------------------------------------------------------------------- *)

type apps = 
  | Apps_tyapp of typ * apps 
  | Apps_app of typ * apps 
  | Apps_done of typ

(* -------------------------------------------------------------------- 
 * ILX extensions to the intruction set
 *
 * The extended array type is only relevant for ILX-style generics.  
 * We erase ty[] to object[] for all array types
 * built with this array constructor.  If 
 * false, it is the normal .NET co-variant array 
 * constructor for the given bounds.  Languages with generics 
 * over arrays should compile their array types with 
 * the flag set to "true".  Note that when the flag is true types such 
 * as "int array" or "string array" will be compiled 
 * to "object[]" and will NOT be compatible with 
 * the corresponding .NET array types.  This effectively places a 
 * language-level restriction on interop when erasing generics, but 
 * this is to be expected when erasing: you only get real polymorphism 
 * types when you have real generics in the type system of the CLR, 
 * i.e. a generic CLR  
 * -------------------------------------------------------------------- *)

type ilx_instr = 
  | EI_lddata of classunion_spec * int * int
  | EI_isdata of classunion_spec * int
  | EI_brisdata of classunion_spec * int * code_label * code_label
  | EI_castdata of bool * classunion_spec * int
  | EI_stdata of classunion_spec * int * int
  | EI_datacase of (bool * classunion_spec * (int * code_label) list * code_label) (* last label is fallthrough, bool is whether to leave value on the stack for each case *)
  | EI_lddatatag of classunion_spec
  | EI_newdata of classunion_spec * int
  | EI_newclo of closure_spec
  | EI_castclo of closure_spec
  | EI_ilzero of typ
  | EI_isclo of closure_spec
  | EI_callclo of tailness * closure_spec * apps
  | EI_stclofld  of (closure_spec * int)  
  | EI_ldenv  of int
  | EI_callfunc of tailness * apps
  | EI_ldftn_then_call of method_spec * (tailness * method_spec * varargs)  (* special: for internal use only *)
  | EI_ld_instance_ftn_then_newobj of method_spec * callsig * (method_spec * varargs)  (* special: for internal use only *)
  | EI_ldelem_any_erasable  of array_shape * typ (* indicates that the array being loaded from *)
                                             (* is a "erasable" array - see above notes on array bounds *)
  | EI_stelem_any_erasable  of array_shape * typ (* see above  *)
  | EI_newarr_erasable      of array_shape * typ (* see above  *)
  | EI_ldlen_multi      of i32 * i32

val mk_ilx_ext_instr: (ilx_instr -> ext_instr)
val is_ilx_ext_instr: (ext_instr -> bool)
val dest_ilx_ext_instr: (ext_instr -> ilx_instr)

val mk_IlxInstr: ilx_instr -> instr

(* -------------------------------------------------------------------- 
 * ILX extensions to the kinds of type definitions available
 * -------------------------------------------------------------------- *)

type closure_info = 
    { cloStructure: lambdas;
      cloFreeVars: freevar list;  
      cloCode: (il_method_body Lazy.t);
      cloSource: source option}

and classunion_info = 
    { cudReprAccess: member_access; (* is the representation public? *)
      cudHelpersAccess: member_access; (* are the representation public? *)
      cudHelpers: bool; (* generate the helpers? *)
      cudAlternatives: alternative array;
      cudNullPermitted: bool;
      cudWhere: source option;  (* debug info for generated code for classunions *) 
    }

type ilx_type_def_kind = 
 | ETypeDef_closure of closure_info
 | ETypeDef_classunion of classunion_info

val mk_ilx_ext_type_def_kind: (ilx_type_def_kind -> ext_type_def_kind)
val is_ilx_ext_type_def_kind: (ext_type_def_kind -> bool)
val dest_ilx_ext_type_def_kind: (ext_type_def_kind -> ilx_type_def_kind)

val mk_ilx_type_def_kind: ilx_type_def_kind -> type_def_kind

(* -------------------------------------------------------------------- 
 * MS-ILX constructs: generalized array types.
 * The boolean flag indicates erasable or not.
 * -------------------------------------------------------------------- *)

val mk_array_ty_old: array_shape * typ -> typ
val gen_mk_array_ty: array_shape * typ * bool -> typ 
val gen_is_array_ty: typ -> bool
val gen_dest_array_ty: typ -> array_shape * typ * bool

(* -------------------------------------------------------------------- 
 * MS-ILX constructs: Closures, thunks, classunions
 * -------------------------------------------------------------------- *)
val inst_apps_aux: int -> genactuals -> apps -> apps
val dest_func_app: apps -> typ * apps
val dest_tyfunc_app: apps -> genactual * apps

val formal_freevar_type_of_cloref : closure_ref -> int -> typ

val tref_of_clospec               : closure_spec -> type_ref
val cloref_of_clospec             : closure_spec -> closure_ref
val formal_freevar_type_of_clospec: closure_spec -> int -> typ
val actual_freevar_type_of_clospec: closure_spec -> int -> typ
val formal_freevars_of_clospec    : closure_spec -> freevar list
val actual_freevars_of_clospec    : closure_spec -> freevar list
val actual_lambdas_of_clospec     : closure_spec -> lambdas
val formal_lambdas_of_clospec     : closure_spec -> lambdas
val inst_of_clospec               : closure_spec -> genactuals

val generalize_cloref: genparam list -> closure_ref -> closure_spec


(* -------------------------------------------------------------------- 
 * MS-ILX: Classunions
 * -------------------------------------------------------------------- *)

val objtype_of_cuspec       : classunion_spec -> typ
val inst_of_cuspec          : classunion_spec -> genactuals
val alts_of_cuspec          : classunion_spec -> alternative list
val altsarray_of_cuspec     : classunion_spec -> alternative array
val tref_of_cuspec          : classunion_spec -> type_ref 
val nullPermitted_of_cuspec : classunion_spec -> bool

val actual_typ_of_cuspec_field: classunion_spec -> int -> int -> typ
val alt_of_cuspec             : classunion_spec -> int -> alternative
val fdef_of_cuspec            : classunion_spec -> int -> int -> field_def

val fdefs_of_alt  : alternative -> field_def array
val fdef_of_alt   : alternative -> int -> field_def 
val name_of_alt   : alternative -> string
val alt_is_nullary: alternative -> bool

val mk_freevar: string * typ -> freevar
val name_of_freevar: freevar -> string 
val typ_of_freevar: freevar -> typ
