(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
module Microsoft.FSharp.Compiler.CcuThunk
open Microsoft.FSharp.Compiler
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
F#*)

open Ast
open Lib
open Printf
open Ildiag

(*---------------------------------------------------------------------------
!* Compilation units and Cross-compilation-unit thunks.
 *
 * A compilation unit is, more or less, the new material created in one
 * invocation of the compiler.  Due to static linking assemblies may hold more 
 * than one compilation unit (i.e. when two assemblies are merged into a compilation
 * the resulting assembly will contain 3 CUs).  Compilation units are also created for referenced
 * .NET assemblies. 
 * 
 * References to items such as type constructors are via 
 * cross-compilation-unit thunks, which directly reference the data structures that define
 * these modules.  Thus, when saving out values to disk we only wish 
 * to save out the "current" part of the term graph.  When reading values
 * back in we "fixup" the links to previously referenced modules.
 *
 * All non-local accesses to the data structures are mediated
 * by ccu-thunks.  Ultimately, a ccu-thunk is either a (named) element of
 * the data structure, or it is a delayed fixup, i.e. an invalid dangling
 * reference that has not had an appropriate fixup applied.  
 *------------------------------------------------------------------------- *)

type 'a ccu_link = 
  | Delayed of string
  | Canonical of 'a ccu_canonical
  | Link of 'a ccu_thunk
and 'a ccu_canonical = { ccu_name: string; mutable ccu_data: 'a } 
and 'a ccu_thunk = { mutable ccu_thunk_contents: 'a ccu_link }
let rec canonical_of_ccu_thunk x = 
  match x.ccu_thunk_contents with
  | Canonical c -> c
  | Link r -> canonical_of_ccu_thunk r
  | Delayed s -> error(Failure("internal error: ccu thunk '"^s^"' not fixed up!"))
let deref_ccu_thunk x = (canonical_of_ccu_thunk x).ccu_data
let rec name_of_ccu_thunk x = 
  match x.ccu_thunk_contents with
  | Canonical c -> c.ccu_name
  | Link r -> name_of_ccu_thunk r
  | Delayed s -> s

let new_ccu_thunk nm x = { ccu_thunk_contents = Canonical {ccu_name = nm; ccu_data = x} }
let new_delayed_ccu_thunk nm = { ccu_thunk_contents = Delayed nm }
let fixup_ccu_thunk reqd avail = 
  match reqd.ccu_thunk_contents with 
  | Canonical c -> errorR(Failure("internal error: ccu thunk "^c.ccu_name^" not delayed!"));
  | Delayed s -> 
      if verbose then dprintf2 "fixup delayed ccu thunk '%s' to '%s'\n" s (name_of_ccu_thunk avail); 
      reqd.ccu_thunk_contents <- Link avail
  | Link t -> errorR(Failure(sprintf "internal error: fixing up link to %s, already pointed to %s" (name_of_ccu_thunk avail) (canonical_of_ccu_thunk t).ccu_name))

let output_ccu_thunk f os thunk = f os (deref_ccu_thunk thunk) 


(*---------------------------------------------------------------------------
!* Fixup pickled data w.r.t. a set of CCU thunks indexed by name
 *------------------------------------------------------------------------- *)

type ('a,'b) pickled_data_with_dangling_references = 
    { ie_raw: 'a; 
      ie_ccus: 'b ccu_thunk list } (* assumptions per reqd. module, starting at top of overall namespace *)

let fixup_ccu_thunks_in_pickled_data loader info =
  List.iter (fun reqd -> fixup_ccu_thunk reqd (loader (name_of_ccu_thunk reqd))) info.ie_ccus;
  info.ie_raw


