//==========================================================================
// (c) Microsoft Corporation 2005-2007.  
//==========================================================================

#light

namespace Microsoft.FSharp.Compiler.Interactive
open Microsoft.FSharp.Core.Pervasives

type IEventLoop =
    interface 
        abstract Run : unit -> bool
        abstract Invoke : (unit -> 'a) -> 'a 
        abstract ScheduleRestart : unit -> unit
    end
    
module SimpleEventLoop = 
    open System.Threading
    open Microsoft.FSharp.Compatibility

    let Create() = 
          let runSignal = new System.Threading.AutoResetEvent(false)
          let exitSignal = new System.Threading.AutoResetEvent(false)
          let doneSignal = new System.Threading.AutoResetEvent(false)
          let queue = ref ([] : (unit -> obj) list)
          let result = ref (None : obj option)
          let setSignal(signal : AutoResetEvent) = while not (signal.Set()) do System.Threading.Thread.Sleep(1); done
          let waitSignal signal = System.Threading.WaitHandle.WaitAll(CompatArray.of_array [| (signal :> WaitHandle) |]) |> ignore
          let waitSignal2 signal1 signal2 = 
              System.Threading.WaitHandle.WaitAny(CompatArray.of_array [| (signal1 :> WaitHandle); (signal2 :> WaitHandle) |])
          let running = ref false
          let restart = ref false
          { new IEventLoop with 
               member x.Run() =  
                   running := true;
                   let rec run() = 
                       match waitSignal2 runSignal exitSignal with 
                       | 0 -> 
                           !queue |> List.iter (fun f -> result := try Some(f()) with _ -> None); 
                           setSignal doneSignal;
                           run()
                       | 1 -> 
                           running := false;
                           !restart
                       | _ -> run()
                   run();
               member x.Invoke(f : unit -> 'a) : 'a  = 
                   queue := [f >> box];
                   setSignal runSignal;
                   waitSignal doneSignal
                   !result |> Option.get |> unbox
               member x.ScheduleRestart() = 
                   // nb. very minor race condition here on running here, but totally 
                   // unproblematic as ScheduleRestart and Exit are almost never called.
                   if !running then 
                       restart := true; 
                       setSignal exitSignal
          }

open Microsoft.FSharp.Text.StructuredFormat.LayoutOps

type A() = 
   class
      let x = 1
      member y.P = x
   end
      
[<Sealed>]
type InteractiveSession() as self = 
    class
        let mutable evLoop = SimpleEventLoop.Create() 
        let mutable showIDictionary = true
        let mutable args = System.Environment.GetCommandLineArgs()
        let mutable opts = { FormatOptions.Default with ShowProperties=true; 
                                                       ShowIEnumerable=true; 
                                                       PrintWidth=78 } 
        // Add a default printer for dictionaries 
        let intercept (ienv: StructuredFormat.IEnvironment) (obj:obj) = 
           match obj with 
           | null -> None 
           | :? System.Collections.IDictionary as ie ->
              let it = ie.GetEnumerator() in 
              let itemLs = 
                  unfoldL // the function to layout each object in the unfold
                          (fun obj -> ienv.GetLayout obj) 
                          // the function to call at each step of the unfold
                          (fun () -> 
                              if it.MoveNext() then 
                                 Some((it.Key, it.Value),()) 
                              else None) () 
                          // the maximum length
                          (1+opts.PrintLength/3) in
              let makeListL itemLs =
                (leftL "[") $$
                sepListL (rightL ";") itemLs $$
                (rightL "]")
              Some(wordL "dict" --- makeListL itemLs)
           | _ -> None 

        let fireThreadExn, threadExn = IEvent.create1<exn>()
        
        do self.PrintIntercepts <-  intercept :: self.PrintIntercepts
        
            
        member self.FloatingPointFormat 
           with get() = opts.FloatingPointFormat
           and set(x) = opts <-  { opts with FloatingPointFormat=x}
        member self.FormatProvider 
           with get() = opts.FormatProvider
           and set(x)= opts <-  { opts with FormatProvider=x}
        member self.PrintWidth  
           with get() = opts.PrintWidth
           and set(x) = opts <-  { opts with PrintWidth=x} 
        member self.PrintDepth  
           with get() = opts.PrintDepth
           and set(x)  = opts <-  { opts with PrintDepth=x}
        member self.PrintLength  
           with get() = opts.PrintLength
           and set(x)  = opts <-  { opts with PrintLength=x} 
        member self.ShowProperties  
           with get() = opts.ShowProperties
           and  set(x) = opts <-  { opts with ShowProperties=x}
        member self.ShowIEnumerable 
           with get() = opts.ShowIEnumerable
           and set(x)  = opts <-  { opts with ShowIEnumerable=x}
        member self.ShowIDictionary
           with get() = showIDictionary
           and set(x)  = showIDictionary <-  x
        member self.PrintIntercepts
           with get() = opts.PrintIntercepts
           and set(x)  = opts <- { opts with PrintIntercepts=x}
        member self.PrintOptions
           with get() = opts
        member self.CommandLineArgs
           with get() = args 
           and set(x)  = args <- x
        member self.AddPrinter(f : 'a -> string) =
          let aty = (type 'a) in 
          let intercept ienv (obj:obj) = 
             match obj with 
             | null -> None 
             | _ when (aty).IsAssignableFrom(obj.GetType())  -> 
                Some(StructuredFormat.LayoutOps.wordL( f(unbox obj))) 
             | _ -> None in 
          opts <-  { opts with PrintIntercepts = (intercept :: opts.PrintIntercepts) }
        member self.EventLoop
           with get() = evLoop
           and set(x)  = evLoop.ScheduleRestart(); evLoop <- x

        member self.AddPrintTransformer(f : 'a -> obj) =
          let aty = (type 'a) in 
          let intercept (ienv:StructuredFormat.IEnvironment)  (obj:obj) = 
             match obj with 
             | null -> None 
             | _ when (aty).IsAssignableFrom(obj.GetType())  -> 
                Some(ienv.GetLayout(f(unbox obj))) 
             | _ -> None in 
          opts <-  { opts with PrintIntercepts = (intercept :: opts.PrintIntercepts) }

        member x.ReportThreadException(exn) = fireThreadExn(exn)
        member x.ThreadException = threadExn
        
    end
  
  
module Settings = 
    let fsi = new InteractiveSession()



module Internals = 
    open System
    open System.Reflection

    let savedIt = ref ((type int),box 0)
    let saveIt (x:'a) = (savedIt := ((type 'a), (box x)))
    let getSavedIt () = snd !savedIt
    let getSavedItType () = fst !savedIt
    let getFsiPrintOptions () = Settings.fsi.PrintOptions
