(* (c) Microsoft Corporation 2005-2007.  *)

(*-------------------------------------------------------------------------
 * Derived expression manipulation and construction functions.
 *------------------------------------------------------------------------- *)

(*F# 
module Microsoft.FSharp.Compiler.Tastops 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Il = Microsoft.Research.AbstractIL.IL 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilx    = Microsoft.Research.AbstractIL.Extensions.ILX 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
F#*)

open Range
open Ast
open Tast
open Env
open Layout

(*-------------------------------------------------------------------------
 * Build common types
 *------------------------------------------------------------------------- *)

val mk_fun_ty : typ -> typ -> typ
val ( --> ) : typ -> typ -> typ
val try_mk_forall_ty : typar_spec list -> typ -> typ
val ( +-> ) : typar_spec list -> typ -> typ
val mk_tuple_ty : typ list -> typ
val mk_iterated_fun_ty : typ list -> typ -> typ
val type_of_lambda_arg : val_spec list -> typ
val mk_multi_lambda_ty : val_spec list -> typ -> typ
val mk_lambda_ty : typar_spec list -> typ list -> typ -> typ

(*-------------------------------------------------------------------------
 * Module publication, used while compiling fslib.
 *------------------------------------------------------------------------- *)

val ensure_fslib_has_submodul_at : ccu -> ident list -> Tast.compilation_path -> xmlDoc -> unit 

(*-------------------------------------------------------------------------
 * Miscellaneous accessors on terms
 *------------------------------------------------------------------------- *)

val strip_expr : expr -> expr

val discrim_of_case : dtree_case -> dtree_discrim
val dest_of_case : dtree_case -> dtree

val var_of_bind : bind -> val_spec
val rhs_of_bind : bind -> expr

(*-------------------------------------------------------------------------
 * Build decision trees imperatively
 *------------------------------------------------------------------------- *)

module MatchBuilder : sig
    type match_builder 
    val create : range -> match_builder
    val add_target : match_builder -> dtree_target -> int
    val add_and_mk_result_target : match_builder -> expr -> dtree
    val close_targets : match_builder -> dtree_target list
    val close : dtree -> match_builder -> range -> typ -> expr
end

(*-------------------------------------------------------------------------
 * Make some special decision graphs
 *------------------------------------------------------------------------- *)

val mk_bool_switch : range -> expr -> dtree -> dtree -> dtree
val mk_cond : range -> typ -> expr -> expr -> expr -> expr
val mk_nonnull_cond : tcGlobals -> range -> typ -> expr -> expr -> expr -> expr

(*-------------------------------------------------------------------------
 * Make normalized references
 *------------------------------------------------------------------------- *)

val mk_local_vref   : val_spec -> val_ref
val mk_local_modref : modul_spec -> modul_ref
val mk_local_tcref  : tycon_spec -> tycon_ref
val mk_local_ecref  : tycon_spec -> tycon_ref

val mk_nonlocal_ccu_top_modref : ccu (* viewed *) -> modul_spec -> modul_ref
val mk_nonlocal_ccu_top_tcref  : ccu (* viewed *) -> tycon_spec -> tycon_ref
val mk_nonlocal_modref  : nonlocal_path -> string -> modul_ref

val mk_modref_in_modref : modul_ref -> modul_spec -> modul_ref
val mk_vref_in_modref   : modul_ref -> val_spec -> val_ref
val mk_ecref_in_modref  : modul_ref -> tycon_spec -> tycon_ref
val mk_tcref_in_modref  : modul_ref -> tycon_spec -> tycon_ref
val mk_tcref_in_tcref   : tycon_ref -> tycon_spec -> tycon_ref
val mk_rfref_in_modref  : modul_ref -> tycon_spec -> ident -> recdfield_ref

(*-------------------------------------------------------------------------
 * Generate new locals
 *------------------------------------------------------------------------- *)

val expr_for_vref : range -> val_ref -> expr

(* NOTE: try to use expr_for_vref or the expression returned from mk_local instead of this. *)
val expr_for_val : range -> val_spec -> expr
val gen_mk_local : range -> string -> typ -> val_mutability -> bool -> val_spec * expr
val mk_local : range -> string -> typ -> val_spec * expr
val mk_compgen_local : range -> string -> typ -> val_spec * expr
val mk_mut_compgen_local : range -> string -> typ -> val_spec * expr
val mk_mut_local : range -> string -> typ -> val_spec * val_ref * expr
val mk_compgen_bind : tcGlobals -> niceNameGenerator -> string -> range -> expr -> val_spec * expr * bind 

(*-------------------------------------------------------------------------
 * Make lambdas
 *------------------------------------------------------------------------- *)

val mk_multi_lambda : range -> val_spec list -> expr * typ -> expr
val mk_basev_multi_lambda : range -> val_spec option -> val_spec list -> expr * typ -> expr
val mk_lambda : range -> val_spec -> expr * typ -> expr
val mk_tlambda : range -> typar_spec list -> expr * typ -> expr
val mk_tchoose : range -> typar_spec list -> expr -> expr
val mk_lambdas : range -> typar_spec list -> val_spec list -> expr * typ -> expr
val mk_multi_lambdas_core : range -> val_spec list list -> expr * typ -> expr * typ
val mk_multi_lambdas : range -> typar_spec list -> val_spec list list -> expr * typ -> expr
val mk_basev_multi_lambdas : range -> typar_spec list -> val_spec option -> val_spec list list -> expr * typ -> expr

val mk_while      : tcGlobals -> expr * expr * range                          -> expr
val mk_for        : tcGlobals -> val_spec * expr * bool * expr * expr * range -> expr
val mk_try_catch  : tcGlobals -> expr * val_spec * expr * val_spec * expr * range * typ         -> expr
val mk_try_finally: tcGlobals -> expr * expr * range * typ                    -> expr

(*-------------------------------------------------------------------------
 * Make let/letrec
 *------------------------------------------------------------------------- *)
 
val mk_multi_lambda_bind : val_spec -> range -> typar_spec list -> val_spec list list -> expr * typ -> bind
val mk_bind : val_spec -> expr -> bind
val mk_binds : val_spec list -> expr list -> bind list
val mk_lets_bind : range -> bind list -> expr -> expr
val mk_let_bind : range -> bind -> expr -> expr
val mk_let : range -> val_spec -> expr -> expr -> expr
val mk_lets : range -> val_spec list -> expr list -> expr -> expr
val mk_let_typed : range -> val_spec -> expr -> expr * 'a -> expr * 'a
val mk_letrec_binds : range -> bind list -> expr -> expr
val mk_letrec_binds_typed : range -> bind list -> expr * 'a -> expr * 'a

 
(*-------------------------------------------------------------------------
 * Type schemes
 *------------------------------------------------------------------------- *)
 
type typ_scheme   = 
    TypeScheme of 
        typars (* the truly generalized type parameters *)
      * typars (* free choice type parameters from a recursive block where this value only generalizes a subsest of the overall set of type parameters generalized *)
      * typ    (* the 'tau' type forming the body of the generalized type *)

val mk_poly_bind_rhs : range -> typ_scheme -> expr -> expr
val is_being_generalized : typar_spec -> typ_scheme -> bool

(*-------------------------------------------------------------------------
 * Make lazy and/or
 *------------------------------------------------------------------------- *)

val mk_lazy_and  : tcGlobals -> range -> expr -> expr -> expr
val mk_lazy_or   : tcGlobals -> range -> expr -> expr -> expr
val mk_byref_typ : tcGlobals -> typ -> typ

(*-------------------------------------------------------------------------
 * Make construction operations
 *------------------------------------------------------------------------- *)

val mk_constr : unionconstr_ref * tinst * expr list * range -> expr
val mk_exnconstr : tycon_ref * expr list * range -> expr
val mk_asm : Il.instr list * tinst * expr list * typ list * range -> expr
val mk_coerce : expr * typ * range * typ -> expr

(*-------------------------------------------------------------------------
 * Make projection operations
 *------------------------------------------------------------------------- *)
 
val mk_tuple_field_get               : expr                   * tinst * int  * range -> expr
val mk_recd_field_get_via_expra      : expr * recdfield_ref   * tinst        * range -> expr
val mk_recd_field_get_addr_via_expra : expr * recdfield_ref   * tinst        * range -> expr
val mk_static_rfield_get              :        recdfield_ref   * tinst        * range -> expr
val mk_static_rfield_set              :        recdfield_ref   * tinst * expr * range -> expr
val mk_static_rfield_get_addr         :        recdfield_ref   * tinst        * range -> expr
val mk_recd_field_set_via_expra      : expr * recdfield_ref   * tinst * expr * range -> expr
val mk_uconstr_tag_get      : expr * tycon_ref       * tinst        * range -> expr
val mk_uconstr_field_get    : expr * unionconstr_ref * tinst * int  * range -> expr
val mk_exnconstr_field_get : expr * tycon_ref               * int  * range -> expr
val mk_uconstr_field_set    : expr * unionconstr_ref * tinst * int  * expr * range -> expr
val mk_exnconstr_field_set : expr * tycon_ref               * int  * expr * range -> expr

(*-------------------------------------------------------------------------
 * Compiled view of tuples (could be in back end?)
 *------------------------------------------------------------------------- *)
 
val maxTuple : int
val compiled_tuple_tcref : tcGlobals -> 'a list -> tycon_ref
val compiled_tuple_ty : tcGlobals -> typ list -> typ
val compiled_mk_tuple : tcGlobals -> typ list * expr list * range -> tycon_ref * typ list * expr list * range
val compiled_get_tuple_field : tcGlobals -> expr * typ list * int * range -> expr * recdfield_ref * typ list * range

(*-------------------------------------------------------------------------
 * Take the address of an expression, or force it into a mutable local. Any allocated
 * mutable local may need to be kept alive over a larger expression, hence we return
 * a wrapping function that wraps "let mutable loc = expr in ..." around a larger
 * expression.
 *------------------------------------------------------------------------- *)

exception PotentialMutationWarning of string * range 
type mutates = DefinitelyMutates | PossiblyMutates | NeverMutates
val mk_expra_of_expr : tcGlobals -> bool -> mutates -> expr -> range -> (expr -> expr) * expr

(*-------------------------------------------------------------------------
 * Tables keyed on values and/or type parameters
 *------------------------------------------------------------------------- *)

type 'a vspec_map 
val vspec_map_find    : val_spec -> 'a vspec_map -> 'a
val vspec_map_tryfind : val_spec -> 'a vspec_map -> 'a option
val vspec_map_mem     : val_spec -> 'a vspec_map -> bool
val vspec_map_add     : val_spec -> 'a -> 'a vspec_map -> 'a vspec_map
val vspec_map_remove  : val_spec -> 'a vspec_map -> 'a vspec_map
val vspec_map_empty   : unit -> 'a vspec_map
val vspec_map_of_list  : (val_spec * 'a) list -> 'a vspec_map

type 'a vspec_mmap = 'a list vspec_map
val vspec_mmap_find : local_val_ref -> 'a vspec_mmap -> 'a list
val vspec_mmap_add  : local_val_ref -> 'a -> 'a vspec_mmap -> 'a vspec_mmap
val vspec_mmap_empty : unit -> 'a vspec_mmap

type 'a typar_map 
val tpmap_find   : typar_spec -> 'a typar_map -> 'a
val tpmap_mem   : typar_spec -> 'a typar_map -> bool
val tpmap_add   : typar_spec -> 'a -> 'a typar_map -> 'a typar_map 
val tpmap_empty : unit -> 'a typar_map 

type 'a tcref_map = TCRefMap of 'a Lib.Imap.t
val tcref_map_tryfind : tycon_ref -> 'a tcref_map -> 'a option
val tcref_map_find    : tycon_ref -> 'a tcref_map -> 'a
val tcref_map_mem     : tycon_ref -> 'a tcref_map -> bool
val tcref_map_add     : tycon_ref -> 'a -> 'a tcref_map -> 'a tcref_map
val tcref_map_empty   : unit -> 'a tcref_map
val tcref_map_of_list : (tycon_ref * 'a) list -> 'a tcref_map

type 'a tcref_mmap = 'a list tcref_map
val tcref_mmap_find : tycon_ref -> 'a tcref_mmap -> 'a list
val tcref_mmap_add  : tycon_ref -> 'a -> 'a tcref_mmap -> 'a tcref_mmap
val tcref_mmap_empty : unit -> 'a tcref_mmap

val val_spec_order : local_val_ref -> local_val_ref -> int
val tycon_spec_order : tycon_spec -> tycon_spec -> int
val rfref_order : recdfield_ref -> recdfield_ref -> int
val typar_spec_order : typar_spec -> typar_spec -> int
val bind_order : bind -> bind -> int

(*-------------------------------------------------------------------------
 * Operations on types: substitution
 *------------------------------------------------------------------------- *)

type typar_inst = (typar_spec * typ) list

type tcref_remap = tycon_ref tcref_map
type tyenv =
    { tpinst : typar_inst;
      tcref_remap : tcref_remap }
val empty_tcref_remap : tcref_remap
val empty_tyenv : tyenv

val mk_typar_inst : typars -> typ list -> typar_inst
val mk_tcref_inst : tycon_ref -> tinst -> typar_inst
val empty_tpinst : typar_inst

val inst_type               : typar_inst -> typ        -> typ
val inst_types              : typar_inst -> tinst      -> tinst
val inst_typar_constraints  : typar_inst -> typar_constraint list -> typar_constraint list 
val inst_trait              : typar_inst -> trait_constraint_info -> trait_constraint_info 
val fixup_typar_constraints : typar_spec -> typar_constraint list -> unit

(* These are just substitution (todo: remove)*)
type tpenv = (typar_spec * typ) list
val empty_tpenv : typar_inst

(*-------------------------------------------------------------------------
 * From typars to types 
 *------------------------------------------------------------------------- *)

val generalize_typars : typars -> tinst
val generalize_tcref : tycon_ref -> typ list * typ
val mk_typar_to_typar_renaming : typars -> typars -> typar_inst * typ list

(*-------------------------------------------------------------------------
 * See through typar equations from inference and/or type abbreviation equations.
 *------------------------------------------------------------------------- *)

val reduce_tcref_abbrev : tycon_ref -> typ list -> typ
val strip_tpeqns_and_tcabbrevsA : bool -> typ -> typ (* set bool to 'true' to allow shortcutting of type parameter equation chains during stripping *)
val strip_tpeqns_and_tcabbrevs : typ -> typ

(*-------------------------------------------------------------------------
 * See through exception abbreviations
 *------------------------------------------------------------------------- *)

val strip_eqns_from_ecref : tycon_ref -> tycon_spec

(*-------------------------------------------------------------------------
 * Analyze types.  These all look through type abbreviations and 
 * inference equations, i.e. are "stripped"
 *------------------------------------------------------------------------- *)

val dest_forall_typ     : (typ -> typars * typ)
val dest_fun_typ        : (typ -> typ * typ)
val dest_tuple_typ      : (typ -> typ list)
val dest_typar_typ      : (typ -> typar_spec)
val try_dest_forall_typ : typ -> typars * typ

val is_fun_ty        : (typ -> bool)
val is_forall_ty     : (typ -> bool)
val is_tuple_ty      : (typ -> bool)
val is_union_ty      : (typ -> bool)
val is_abstract_ty   : (typ -> bool)
val is_fsobjmodel_ty : (typ -> bool)
val is_recd_ty       : (typ -> bool)
val is_typar_ty      : (typ -> bool)

val mk_tyapp_ty : tycon_ref -> tinst -> typ

val is_stripped_tyapp_typ    : typ -> bool
val dest_stripped_tyapp_typ  : typ -> tycon_ref * tinst
val tcref_of_stripped_typ    : typ -> tycon_ref
val tinst_of_stripped_typ    : typ -> tinst
val mk_inst_for_stripped_typ : typ -> typar_inst

val domain_of_fun_typ  : typ -> typ
val range_of_fun_typ   : typ -> typ
val strip_fun_typ      : typ -> typ list * typ
val strip_fun_typ_upto : int -> typ -> typ list * typ

val reduce_forall_typ : typ -> tinst -> typ

val dest_tuple_typ_upto : int -> typ -> typ list
val try_dest_tuple_typ : typ -> typ list

(* union data constructors *)
val rty_of_ucref               : unionconstr_ref -> typ
val rfields_of_ucref           : unionconstr_ref -> recdfield_spec list
val rty_of_uctyp : unionconstr_ref -> tinst -> typ
val ucref_index: unionconstr_ref -> int
val ucrefs_of_tcref: tycon_ref -> unionconstr_ref list

(* fields of union data constructors *)
val typ_of_ucref_rfield_by_idx : unionconstr_ref -> tinst -> int -> typ
val typs_of_ucref_rfields : typar_inst -> unionconstr_ref -> typ list

(* fields of records *)
val formal_typ_of_rfield    :               recdfield_spec -> typ
val literal_value_of_rfield :               recdfield_spec -> tconst option
val zero_init_of_rfield  :               recdfield_spec -> bool
val typ_of_rfield           : typar_inst -> recdfield_spec -> typ

val static_of_rfref        : recdfield_ref -> bool
val literal_value_of_rfref : recdfield_ref -> tconst option
val actual_rtyp_of_rfref   : recdfield_ref -> tinst -> typ
val rfref_index            : recdfield_ref -> int

val typs_of_tcref_rfields    : typar_inst -> tycon_ref -> typ list
val actual_typ_of_rfield    : tycon_spec -> tinst -> recdfield_spec -> typ
val instance_rfrefs_of_tcref         : tycon_ref -> recdfield_ref list
val all_rfrefs_of_tcref     : tycon_ref -> recdfield_ref list

(* fields of exception constructors *)
val typs_of_ecref_rfields : tycon_ref -> typ list

val iterType : (typ -> unit) * (local_typar_ref -> unit) -> typ -> unit

(*-------------------------------------------------------------------------
 * Top types: guaranteed to be compiled to .NET methods, and must be able to 
 * have user-specified argument names (for stability w.r.t. reflection)
 * and user-specified argument and return attributes.
 *------------------------------------------------------------------------- *)

val dest_top_tau_type : topArgInfo list list -> typ -> (typ * topArgInfo) list list * typ
val dest_top_type     : val_arity -> typ -> typars * (typ * topArgInfo) list list * typ * topArgInfo

val normalizeDeclaredtyparsForEquiRecursiveInference : typars -> typars

(*-------------------------------------------------------------------------
 * Compute the return type after an application
 *------------------------------------------------------------------------- *)
 
val apply_types : typ -> typ list * 'a list -> typ

(*-------------------------------------------------------------------------
 * Compute free variables in types
 *------------------------------------------------------------------------- *)
 
val empty_free_loctypars : free_loctypars
val union_free_loctypars : free_loctypars -> free_loctypars -> free_loctypars

val empty_free_loctycons : free_loctycons
val union_free_loctycons : free_loctycons -> free_loctycons -> free_loctycons

val empty_free_tyvars : free_tyvars
val union_free_tyvars : free_tyvars -> free_tyvars -> free_tyvars

val acc_free_in_type : typ -> free_tyvars -> free_tyvars
val acc_free_tprefs : typars -> free_tyvars -> free_tyvars

val free_in_type  : typ      -> free_tyvars
val free_in_types : typ list -> free_tyvars
val free_in_val   : val_spec -> free_tyvars

(* This one puts free variables in canonical left-to-right order. *)
val free_in_type_lr : bool -> typ -> typars
val free_in_types_lr : bool -> typ list -> typars
val free_in_types_lr_no_cxs : typ list -> typars


(*-------------------------------------------------------------------------
 * Equivalence of types (up to substitution of type variables in the left-hand type)
 *------------------------------------------------------------------------- *)

type type_equiv_env = 
    { ae_typars: typ typar_map;
      ae_tcrefs: tcref_remap }

val tyeq_env_empty : type_equiv_env

val bind_tyeq_env_typars : typars -> typars   -> type_equiv_env -> type_equiv_env
val bind_tyeq_env_types  : typars -> typ list -> type_equiv_env -> type_equiv_env
val bind_tyeq_env_tpinst : typar_inst         -> type_equiv_env -> type_equiv_env
val mk_tyeq_env          : typars -> typars                     -> type_equiv_env

val traits_aequiv           : tcGlobals -> type_equiv_env -> trait_constraint_info -> trait_constraint_info -> bool
val typarConstraints_aequiv : tcGlobals -> type_equiv_env -> typar_constraint      -> typar_constraint      -> bool
val typar_decls_aequiv           : tcGlobals -> type_equiv_env -> typars                -> typars                -> bool
val type_aequiv             : tcGlobals -> type_equiv_env -> typ                   -> typ                   -> bool
val tcref_aequiv            : tcGlobals -> type_equiv_env -> tycon_ref             -> tycon_ref             -> bool
val type_equiv              : tcGlobals                   -> typ                   -> typ                   -> bool

(*-------------------------------------------------------------------------
 * VSPR values (i.e. values that are really members in the object model)
 *------------------------------------------------------------------------- *)

val member_vref_is_instance : val_ref -> bool
val member_val_is_instance  : val_spec -> bool
val dest_member_vref_typ : tcGlobals -> val_ref -> typars * (typ * topArgInfo) list * typ * topArgInfo
val partition_val_typars : val_spec -> (typars * typars * typars * typar_inst * typ list) option
val partition_vref_typars : val_ref -> (typars * typars * typars * typar_inst * typ list) option

val logical_name_of_member : val_meminfo -> string
val chop_property_name : string -> string
val get_property_name : val_meminfo -> string

val vtyp_of_property_val : tcGlobals -> val_spec -> typ
val arginfos_of_propery_val : tcGlobals -> val_spec -> (typ * topArgInfo) list 
val arginfos_of_member_val: tcGlobals -> val_spec -> (typ * topArgInfo) list 
val arginfos_of_member_vref: tcGlobals -> val_ref -> (typ * topArgInfo) list 

val mustinline : val_inline_info -> bool

val get_member_call_info : tcGlobals -> val_ref * val_use_flag -> int * bool * bool * bool * bool * bool * bool * bool

(*-------------------------------------------------------------------------
 * Printing
 *------------------------------------------------------------------------- *)
 
module PrettyTypes : sig
    val prettyTyparNames : (typar_spec -> bool) -> char -> string list -> typars -> string list
    val prettify1 : typ -> typar_inst * typ * (typar_spec * typar_constraint) list
    val prettify2 : typ * typ -> typar_inst * (typ * typ) * (typar_spec * typar_constraint) list
    val prettifyN : typ list -> typar_inst * typ list * (typar_spec * typar_constraint) list

end

type displayEnv = {html: bool;
                   htmlHideRedundantKeywords: bool;
                   htmlAssemMap: string Lib.namemap;
                   openTopPaths: (string list) list;
                   showTyparBinding: bool;
                   showImperativeTyparAnnotations: bool;
                   showMemberContainers: bool;
                   showAttributes: bool;
                   showConstraintTyparAnnotations:bool;
                   showTyparDefaultConstraints: bool;
                   g: tcGlobals }
val empty_denv : tcGlobals -> displayEnv
val denv_add_open_path   : path   -> displayEnv -> displayEnv
val denv_add_open_modref : modul_ref   -> displayEnv -> displayEnv

val demangle_operator : string -> string
val demangle_dotnet_generic_overloading : string -> string
val display_name_of_val : val_spec -> string
val display_name_of_vref : val_ref -> string
val compiled_name_of_val : val_spec -> string
val display_name_of_typar : typar_spec -> string

val full_name_of_nlpath : nonlocal_path -> string
val full_name_of_modref : modul_ref -> string
val full_name_of_vref   : val_ref -> string
val full_name_of_tcref  : tycon_ref -> string
val full_name_of_ecref  : tycon_ref -> string
val full_name_of_ucref  : unionconstr_ref -> string
val full_name_of_rfref  : recdfield_ref -> string
val qualified_name_of_tcref : tycon_ref -> string -> string

val display_name_of_tycon : tycon_spec -> string
val display_name_of_tcref : tycon_ref -> string

module NicePrint : sig
    val tconstL : tconst -> layout
    val typL                        : displayEnv -> typ -> layout
    val constraintL                 : displayEnv -> (typar_spec * typar_constraint) -> layout 
    val topTypAndConstraintsL       : displayEnv -> (typ * topArgInfo) list -> typ -> layout
    val typesAndConstraintsL        : displayEnv -> typ list -> layout list * layout
    val tyconSpecL                  : bool -> displayEnv -> layout -> tycon_spec -> layout
    val memberSigL                  : displayEnv -> typar_inst * string * typars * (typ * topArgInfo) list * typ -> layout
    val valSpecL                    : displayEnv -> val_spec -> layout
    val exnSpecL                    : displayEnv -> tycon_spec -> layout
    val exnSpecReprL                : displayEnv -> tycon_exnc_info -> layout
    val inferred_sig_of_structL     : bool -> displayEnv -> modul_expr -> layout
    val modul_specL                 : displayEnv -> modul_typ -> layout
    val modulL                      : displayEnv -> modul_spec -> layout
    
    val dataExprL                   : displayEnv -> expr -> layout
    
    val string_of_typ               : displayEnv -> typ -> string
    val pretty_string_of_typ        : displayEnv -> typ -> string
    val string_of_typar_constraints : displayEnv -> (typar_spec * typar_constraint) list  -> string

    val output_tref              : displayEnv -> Buffer.t -> Il.type_ref -> unit
    val output_tcref             : displayEnv -> Buffer.t -> tycon_ref -> unit
    val output_typ               : displayEnv -> Buffer.t -> typ -> unit
    val output_typars            : displayEnv -> string -> Buffer.t -> typars -> unit
    val output_typar_constraint  : displayEnv -> Buffer.t -> typar_spec * typar_constraint -> unit
    val output_typar_constraints : displayEnv -> Buffer.t -> (typar_spec * typar_constraint) list -> unit
    val output_qualified_val_spec: displayEnv -> Buffer.t -> val_spec -> unit

    val output_tycon        : bool -> displayEnv -> Buffer.t -> tycon_spec -> unit
    val output_uconstr  : displayEnv -> Buffer.t -> unionconstr_spec -> unit
    val output_rfield   : displayEnv -> Buffer.t -> recdfield_spec -> unit
    val output_exnc     : displayEnv -> Buffer.t -> tycon_spec -> unit

end
val adhoc_of_tycon : tycon_spec -> val_ref list
val super_of_tycon : tcGlobals -> tycon_spec -> Tast.typ
val implements_of_tycon : tcGlobals -> tycon_spec -> Tast.typ list
val vslot_vals_of_tycons : tycon_spec list -> val_spec list

(*-------------------------------------------------------------------------
 * Free variables in expressions etc.
 *------------------------------------------------------------------------- *)

val empty_freevars : freevars
val union_freevars : freevars -> freevars -> freevars

val acc_free_in_targets : dtree_target array -> freevars -> freevars
val acc_free_in_exprs : expr list -> freevars -> freevars
val acc_free_in_switch_cases : dtree_case list -> dtree option -> freevars -> freevars

val free_in_mdef : modul_def -> freevars
val free_in_expr  : expr  -> freevars
val free_in_rhs   : bind  -> freevars

(*-------------------------------------------------------------------------
 * Mark/range/position information from expressions
 *------------------------------------------------------------------------- *)

val range_of_expr : expr -> range

(*-------------------------------------------------------------------------
 * Top expressions to implement top types
 *------------------------------------------------------------------------- *)

val dest_top_lambda : expr * typ -> typars * val_spec list list * expr * typ
val infer_arity_of_expr : typ -> attribs list list -> attribs -> expr -> val_arity
val infer_arity_of_expr_bind : val_spec -> expr -> val_arity
val chosen_arity_of_bind : bind -> val_arity_info

(*-------------------------------------------------------------------------
 *  Copy expressions and types
 *------------------------------------------------------------------------- *)

type vspec_remap = val_ref vspec_map
type expr_remap = { vspec_remap: vspec_remap;
                    tyenv: tyenv }

val empty_expr_remap : expr_remap

(* REVIEW: this mutation hack should not be used *)
val set_val_has_no_arity : val_spec -> val_spec

val remap_tcref : tcref_remap -> tycon_ref -> tycon_ref
val remap_ucref : tcref_remap -> unionconstr_ref -> unionconstr_ref
val remap_rfref : tcref_remap -> recdfield_ref -> recdfield_ref
val remap_vref : expr_remap -> val_ref -> val_ref
val remap_expr : tcGlobals -> bool -> expr_remap -> expr -> expr
val remap_type : tyenv -> typ -> typ
val copy_val : tcGlobals -> bool -> val_spec -> val_spec
val copy_mtyp : tcGlobals -> bool -> modul_typ -> modul_typ
val copy_expr : tcGlobals -> bool -> expr -> expr
val copy_assembly : tcGlobals -> bool -> typedAssembly -> typedAssembly
val copy_slotsig : tslotsig -> tslotsig
val inst_slotsig : typar_inst -> tslotsig -> tslotsig
val inst_expr : tcGlobals -> typar_inst -> expr -> expr

(*-------------------------------------------------------------------------
 * Build the remapping that corresponds to a module meeting its signature
 * and also report the set of tycons, tycon representations and values hidden in the process.
 *------------------------------------------------------------------------- *)

type module_repackage_information = 
    { mrpiVals: (val_ref * val_ref) list;
      mrpiTycons: (tycon_ref * tycon_ref) list  }
      
type module_hiding_information = 
    { mhiTycons  : tycon_spec Zset.t; 
      mhiTyconReprs : tycon_spec Zset.t;  
      mhiVals       : val_spec Zset.t; 
      mhiRecdFields : recdfield_ref Zset.t;
      mhiUnionConstrs : unionconstr_ref Zset.t }

val empty_mrpi : module_repackage_information
val empty_mhi : module_hiding_information
val union_mhi : module_hiding_information -> module_hiding_information -> module_hiding_information

val mk_mtyp_to_mtyp_remapping : (modul_typ -> modul_typ -> module_repackage_information * module_hiding_information)
val mk_mdef_to_mtyp_remapping : (modul_def -> modul_typ -> module_repackage_information * module_hiding_information)
val mk_assembly_boundary_mhi : (modul_typ -> module_hiding_information)
val mtyp_of_mexpr : modul_expr -> modul_typ
val mk_repackage_remapping : module_repackage_information -> expr_remap 

val wrap_modul_in_namespace : ident -> modul_spec -> modul_spec
val wrap_mbind_in_namespace : ident -> modul_bind -> modul_bind
val wrap_modul_as_mtyp_in_namespace : modul_spec -> modul_typ 
val wrap_mtyp_as_mspec : ident -> compilation_path -> modul_typ -> modul_spec

val sigTypeOfImplFile : typedImplFile -> modul_typ

(*-------------------------------------------------------------------------
 * Given a list of top-most signatures that together constrain the public compilation units
 * of an assembly, compute a remapping that converts local references to non-local references.
 * This remapping must be applied to all pickled expressions and types 
 * exported from the assembly.
 *------------------------------------------------------------------------- *)

val mk_export_remapping : ccu -> (modul_spec -> expr_remap)
val apply_export_remapping_to_mspec :  tcGlobals -> expr_remap -> modul_spec -> modul_spec 

(*-------------------------------------------------------------------------
 *  Adjust marks in expressions
 *------------------------------------------------------------------------- *)

val remark_expr : range -> expr -> expr

(*-------------------------------------------------------------------------
 *  Make applications
 *------------------------------------------------------------------------- *)
 
val prim_mk_app : (expr * typ) -> tinst -> expr list -> range -> expr
val mk_appl : (expr * typ) * typ list list * expr list * range -> expr
val mk_tyapp : range -> expr * typ -> typ list -> expr

val mk_val_set   : range -> val_ref -> expr -> expr
val mk_lval_set  : range -> val_ref -> expr -> expr
val mk_lval_get  : range -> val_ref -> expr
val mk_val_addr  : range -> val_ref -> expr

(*-------------------------------------------------------------------------
 * Note these take the address of the record expression if it is a struct, and
 * apply a type instantiation if it is a first-class polymorphic record field.
 *------------------------------------------------------------------------- *)

val mk_recd_field_get : tcGlobals -> expr * recdfield_ref * tinst * tinst * range -> expr
val mk_recd_field_set : tcGlobals -> expr * recdfield_ref * tinst * expr * range -> expr

(*-------------------------------------------------------------------------
 *  Get the targets used in a decision graph (for reporting warnings)
 *------------------------------------------------------------------------- *)

val acc_targets_of_dtree : dtree -> int list -> int list

(*-------------------------------------------------------------------------
 *  Optimizations on decision graphs
 *------------------------------------------------------------------------- *)

val mk_and_optimize_match : range -> range -> typ -> dtree -> dtree_target list -> expr

(*-------------------------------------------------------------------------
 *  Work out what things on the r.h.s. of a letrec need to be fixed up
 *------------------------------------------------------------------------- *)

val iter_letrec_fixups : 
   tcGlobals -> val_spec option  -> 
   (val_spec option -> expr -> (expr -> expr) -> expr -> unit) -> 
   expr * (expr -> expr) -> expr -> unit

(*-------------------------------------------------------------------------
 * From lambdas taking multiple variables to lambdas taking a single variable
 * of tuple type. 
 *------------------------------------------------------------------------- *)

val multi_lambda_to_tupled_lambda: niceNameGenerator -> val_spec list -> expr -> val_spec * expr
val adjust_arity_of_lambda_body          : tcGlobals -> niceNameGenerator -> int -> val_spec list -> expr -> val_spec list * expr

(*-------------------------------------------------------------------------
 * Make applications, doing beta reduction by introducing let-bindings
 *------------------------------------------------------------------------- *)

val beta_mk_appl : tcGlobals -> niceNameGenerator -> expr * typ * tinst list * expr list * range -> expr

(*-------------------------------------------------------------------------
 * Equality on some items
 *------------------------------------------------------------------------- *)

val const_eq : tconst -> tconst -> bool
val discrim_eq : tcGlobals -> dtree_discrim -> dtree_discrim -> bool

val joinTyparStaticReq : typarStaticReq -> typarStaticReq -> typarStaticReq

(*-------------------------------------------------------------------------
 * More layout - this is for debugging
 *------------------------------------------------------------------------- *)
module DebugPrint : sig

    val layout_stamps : bool ref
    val layout_ranges : bool ref
    val showType : typ -> string
    val showExpr : expr -> string

end

val vrefL : val_ref -> layout
val ucrefL : unionconstr_ref -> layout
val vspecAtBindL : val_spec -> layout
val intL : int -> layout
val vspecL : val_spec -> layout
val typarDeclL : typar_spec -> layout
val traitL : trait_constraint_info -> layout
val typarL : typar_spec -> layout
val typarsL : typars -> layout
val typeL : typ -> layout
val tslotsigL : tslotsig -> layout
val mtypeL : modul_typ -> layout
val mspecL : modul_spec -> layout
val vspecTyL : val_spec -> layout
val vsprL : val_meminfo -> layout
val bindL : bind -> layout
val exprL : expr -> layout
val tyconL : tycon_spec -> layout
val dtreeL : dtree -> layout
val mimplsL : typedAssembly -> layout
val rfrefL : recdfield_ref -> layout



(*-------------------------------------------------------------------------
 * Fold on expressions
 *------------------------------------------------------------------------- *)

type 'a exprFolder =
    { exprIntercept: ('a -> expr -> 'a) -> 'a -> expr -> 'a option;
      valDefAcc: 'a -> bool * val_spec -> 'a;
      valBindAcc         : 'a -> bool * bind list -> 'a;         
      dtreeAcc: 'a -> dtree -> 'a;
      targetIntercept: ('a -> expr -> 'a) -> 'a -> dtree_target -> 'a option;
      tmethodIntercept: ('a -> expr -> 'a) -> 'a -> tmethod -> 'a option;}
val exprFolder0 : 'a exprFolder
val foldAssembly : 'a exprFolder -> ('a -> typedAssembly -> 'a) 
val foldExpr : 'a exprFolder -> ('a -> expr -> 'a) 
val exprStats : expr -> string

(*-------------------------------------------------------------------------
 * Make some common types
 *------------------------------------------------------------------------- *)

val mk_nativeptr_ty  : tcGlobals -> typ -> typ
val mk_array_ty      : tcGlobals -> typ -> typ
val is_option_ty     : tcGlobals -> typ -> bool
val dest_option_ty   : tcGlobals -> typ -> typ

(*-------------------------------------------------------------------------
 * Primitives associated with compiling the IEvent idiom to .NET events
 *------------------------------------------------------------------------- *)

val is_fslib_IPrimitiveDelegateEvent_ty   : tcGlobals -> typ -> bool
val dest_fslib_IPrimitiveDelegateEvent_ty : tcGlobals -> typ -> typ 
val mk_fslib_IDelegateEvent_ty   : tcGlobals -> typ -> typ -> typ
val mk_fslib_IEvent_ty           : tcGlobals -> typ -> typ

(*-------------------------------------------------------------------------
 * Primitives associated with printf format string parsing
 *------------------------------------------------------------------------- *)

val mk_lazy_ty : tcGlobals -> typ -> typ
val mk_lazystatus_ty : tcGlobals -> typ -> typ
val mk_format_ty : tcGlobals -> typ -> typ -> typ -> typ -> typ -> typ

(*-------------------------------------------------------------------------
 * Classify types
 *------------------------------------------------------------------------- *)

val is_il_named_typ      : typ -> bool
val mk_il_arr_ty         : tcGlobals -> int -> typ -> typ
val is_il_arr_tcref      : tcGlobals -> tycon_ref -> bool
val rank_of_il_arr_tcref : tcGlobals -> tycon_ref -> int
val is_il_arr_typ        : tcGlobals -> typ -> bool
val is_il_arr1_typ       : tcGlobals -> typ -> bool
val dest_il_arr1_typ     : tcGlobals -> typ -> typ
val is_compat_array_typ  : tcGlobals -> typ -> bool
val is_unit_typ          : tcGlobals -> typ -> bool
val is_il_ref_typ        : tcGlobals -> typ -> bool

val is_any_array_typ      : tcGlobals -> typ -> bool
val dest_any_array_typ    : tcGlobals -> typ -> typ
val rank_of_any_array_typ : tcGlobals -> typ -> int

val is_fsobjmodel_ref_typ         : typ -> bool
val is_fsobjmodel_struct_tycon    : tycon_spec -> bool
val is_fsobjmodel_enum_tycon      : tycon_spec -> bool
val is_struct_tycon               : tycon_spec -> bool
val is_struct_tcref               : tycon_ref -> bool
val is_enum_tycon                 : tycon_spec -> bool
val is_enum_tcref                 : tycon_ref -> bool
val is_interface_tycon                 : tycon_spec -> bool
val is_interface_tcref                 : tycon_ref -> bool
val is_fsobjmodel_interface_tycon : tycon_spec -> bool
val is_fsobjmodel_delegate_tycon  : tycon_spec -> bool
val is_delegate_typ  : typ -> bool
val is_interface_typ : typ -> bool
val is_ref_typ    : tcGlobals -> typ -> bool
val is_sealed_typ : tcGlobals -> typ -> bool
val underlying_typ_of_enum_typ : tcGlobals -> typ -> typ
val is_struct_typ : typ -> bool
val is_class_typ  : typ -> bool
val is_enum_typ   : typ -> bool
val is_flag_enum_typ   : tcGlobals -> typ -> bool

(*-------------------------------------------------------------------------
 * Special semantic constraints
 *------------------------------------------------------------------------- *)

val isUnionThatUsesNullAsRepresentation: tcGlobals -> tycon_spec -> bool
val vsprCompiledAsInstance : tcGlobals -> tycon_ref -> bool -> val_meminfo -> Tast.attrib list -> bool
val valCompiledAsInstance : tcGlobals -> val_spec -> bool
val vrefCompiledAsInstance : tcGlobals -> val_ref -> bool
val moduleNameIsMangled : tcGlobals -> Tast.attrib list -> bool

val typNullIsExtraValue : tcGlobals -> typ -> bool
val typNullIsTrueValue : tcGlobals -> typ -> bool
val typNullNotLiked : tcGlobals -> typ -> bool
val typNullNever : tcGlobals -> typ -> bool

val typSatisfiesNullConstraint : tcGlobals -> typ -> bool

val is_partially_implemented_tycon : tycon_spec -> bool

val ucref_alloc_observable : unionconstr_ref -> bool
val tycon_alloc_observable : tycon_spec -> bool
val tcref_alloc_observable : tycon_ref -> bool
val ecref_alloc_observable : tycon_ref -> bool 
val ucref_rfield_mutable : tcGlobals -> unionconstr_ref -> int -> bool
val ecref_rfield_mutable : tycon_ref -> int -> bool

val use_genuine_field : tycon_spec -> recdfield_spec -> bool 
val gen_field_name : tycon_spec -> recdfield_spec -> string

(*-------------------------------------------------------------------------
 * Destruct slotsigs etc.
 *------------------------------------------------------------------------- *)

val slotsig_has_void_rty     : tcGlobals ->      tslotsig -> bool
val actual_rty_of_slotsig    : tinst -> tinst -> tslotsig -> typ
val enclosing_typ_of_slotsig :                   tslotsig -> typ
val name_of_slotsig          :                   tslotsig -> string

val rty_of_tmethod : tmethod -> typ
val id_of_method   : tmethod -> ident

(*-------------------------------------------------------------------------
 * Primitives associated with initialization graphs
 *------------------------------------------------------------------------- *)

val mk_refcell              : tcGlobals -> range -> typ -> expr -> expr
val mk_refcell_get          : tcGlobals -> range -> typ -> expr -> expr
val mk_refcell_set          : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_lazy_delayed         : tcGlobals -> range -> typ -> expr -> expr
val mk_lazystatus_undefined : tcGlobals -> range -> typ -> expr
val mk_lazy_force           : tcGlobals -> range -> typ -> val_spec -> expr -> expr

val mk_refcell_contents_rfref : tcGlobals -> recdfield_ref
val is_refcell_ty   : tcGlobals -> typ -> bool
val dest_refcell_ty : tcGlobals -> typ -> typ
val mk_refcell_ty   : tcGlobals -> typ -> typ

val mk_seq_ty    : tcGlobals -> typ -> typ
val mk_list_ty   : tcGlobals -> typ -> typ
val mk_option_ty : tcGlobals -> typ -> typ
val mk_none_ucref : tcGlobals -> unionconstr_ref
val mk_some_ucref : tcGlobals -> unionconstr_ref

val mk_nil  : tcGlobals -> range -> typ -> expr
val mk_cons : tcGlobals -> typ -> expr -> expr -> expr

(*-------------------------------------------------------------------------
 * Generate IL sequence points (with caching)
 *------------------------------------------------------------------------- *)

val normalize : string -> string
val gen_range : range -> Il.source option

(*-------------------------------------------------------------------------
 * Make a few more expressions
 *------------------------------------------------------------------------- *)

val mk_seq  : range -> expr -> expr -> expr
val mk_seqs : tcGlobals -> range -> expr list -> expr   
val mk_recd : recordConstructionInfo * tycon_ref * tinst * recdfield_ref list * expr list * range -> expr
val mk_unbox : typ -> expr -> range -> expr
val mk_isinst : typ -> expr -> range -> expr
val mk_null : range -> typ -> expr
val mk_nonnull_test : tcGlobals -> range -> expr -> expr
val mk_nonnull_poke : tcGlobals -> range -> expr -> expr
val mk_isinst_cond : tcGlobals -> range -> typ -> (val_spec * expr) -> val_spec -> expr -> expr -> expr
val mk_throw   : range -> typ -> expr -> expr
val mk_rethrow : range -> typ -> expr -> expr
val mk_ldarg0 : range -> typ -> expr

val mk_ilzero : range * typ -> expr

val mk_string    : tcGlobals -> range -> Bytes.bytes -> expr
val mk_int64     : tcGlobals -> range -> int64 -> expr
val mk_bool      : tcGlobals -> range -> bool -> expr
val mk_true      : tcGlobals -> range -> expr
val mk_false     : tcGlobals -> range -> expr
val mk_unit      : tcGlobals -> range -> expr
val mk_int32     : tcGlobals -> range -> int32 -> expr
val mk_int       : tcGlobals -> range -> int -> expr
val mk_zero      : tcGlobals -> range -> expr
val mk_one       : tcGlobals -> range -> expr
val mk_minus_one : tcGlobals -> range -> expr
val dest_int32 : expr -> int32 option

(*-------------------------------------------------------------------------
 * Primitives associated with quotations
 *------------------------------------------------------------------------- *)
 
val mk_expr_ty : tcGlobals -> typ -> typ
val mk_expr_template_ty : tcGlobals -> typ -> typ -> typ -> typ -> typ
val mk_raw_expr_ty : tcGlobals -> typ
val mk_raw_expr_template_ty : tcGlobals -> typ -> typ -> typ
val mspec_Type_GetTypeFromHandle : Il.mscorlib_refs ->  Il.method_spec
val fspec_Missing_Value : Il.mscorlib_refs ->  Il.field_spec
val mk_bytearray_ty : tcGlobals -> typ

(*-------------------------------------------------------------------------
 * Helpers associated with code-generation of comparison/hash augmentations
 *------------------------------------------------------------------------- *)

val mk_hash_param_ty : tcGlobals -> typ -> typ
val mk_compare_ty    : tcGlobals -> typ -> typ
val mk_equals_obj_ty : tcGlobals -> typ -> typ
val mk_rel_ty : tcGlobals -> typ -> typ

(*-------------------------------------------------------------------------
 * Construct calls to some intrinsic functions
 *------------------------------------------------------------------------- *)

val mk_call_string_compare              : tcGlobals -> range -> expr -> expr -> expr
val mk_call_new_format              : tcGlobals -> range -> typ -> typ -> typ -> typ -> typ -> expr -> expr

val mk_call_unbox       : tcGlobals -> range -> typ -> expr -> expr
val mk_call_unbox_fast  : tcGlobals -> range -> typ -> expr -> expr
val can_use_unbox_fast : tcGlobals -> typ -> bool

val mk_call_istype      : tcGlobals -> range -> typ -> expr -> expr
val mk_call_istype_fast : tcGlobals -> range -> typ -> expr -> expr
val can_use_istype_fast : tcGlobals -> typ -> bool

val mk_call_typeof      : tcGlobals -> range -> typ -> expr 

val mk_call_create_instance             : tcGlobals -> range -> typ -> expr
val mk_call_poly_compare_outer          : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_poly_equals_outer           : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_poly_hash_param_outer       : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_array_get                   : tcGlobals -> range -> typ -> expr -> expr -> expr

val mk_call_string_to_bigint_vref       : tcGlobals -> range -> expr -> expr
val mk_call_string_to_bignum_vref       : tcGlobals -> range -> expr -> expr
val try_elim_bigint_bignum_constants : tcGlobals -> range -> tconst -> expr option

val mk_call_unpickle_quotation_vref     : tcGlobals -> range -> typ list -> expr -> expr -> expr
val mk_call_unpickle_raw_quotation_vref : tcGlobals -> range -> typ list -> expr -> expr -> expr
val mk_call_seq_map_concat      : tcGlobals -> range -> typ  -> typ -> expr -> expr -> expr
val mk_call_seq_using           : tcGlobals -> range -> typ  -> typ -> expr -> expr -> expr
val mk_call_seq_delay           : tcGlobals -> range -> typ  -> expr -> expr
val mk_call_seq_append          : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_seq_finally         : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_seq_generated       : tcGlobals -> range -> typ -> expr -> expr -> expr
val mk_call_seq_of_functions    : tcGlobals -> range -> typ  -> typ -> expr -> expr -> expr -> expr
val mk_call_seq_to_array        : tcGlobals -> range -> typ  -> expr -> expr 
val mk_call_seq_to_list         : tcGlobals -> range -> typ  -> expr -> expr 
val mk_call_seq_map             : tcGlobals -> range -> typ  -> typ -> expr -> expr -> expr
val mk_call_seq_singleton       : tcGlobals -> range -> typ  -> expr -> expr
val mk_call_seq_empty           : tcGlobals -> range -> typ  -> expr
val mk_ceq                      : tcGlobals -> range -> expr -> expr -> expr

val mk_case : dtree_discrim * dtree -> dtree_case

(*-------------------------------------------------------------------------
 * operations primarily associated with the optimization to fix
 * up loops to generate .NET code that does not include array bound checks
 *------------------------------------------------------------------------- *)

val dest_incr : expr -> expr option
val mk_decr   : tcGlobals -> range -> expr -> expr
val mk_incr   : tcGlobals -> range -> expr -> expr
val mk_ldlen  : tcGlobals -> range -> expr -> expr

(*-------------------------------------------------------------------------
 * type-of operations on the expression tree
 *------------------------------------------------------------------------- *)

val type_of_expr : tcGlobals -> expr -> typ 

(*-------------------------------------------------------------------------
 * Analyze attribute sets 
 *------------------------------------------------------------------------- *)

val ilthing_has_il_attrib : Il.type_ref -> Il.custom_attrs -> bool
val ilthing_decode_il_attrib : tcGlobals -> Il.type_ref -> Il.custom_attrs -> (Il.custom_attr_elem list * Il.custom_attr_named_arg list) option
val is_fs_attrib             : tcGlobals -> Env.attrib_info -> attrib -> bool
val fsthing_has_attrib       : tcGlobals -> Env.attrib_info -> attrib list -> bool
val fsthing_tryfind_attrib   : tcGlobals -> Env.attrib_info -> attrib list -> attrib option
val fsthing_unit_attrib      : tcGlobals -> Env.attrib_info -> attrib list -> unit option
val fsthing_bool_attrib      : tcGlobals -> Env.attrib_info -> attrib list -> bool option
val fsthing_int32_attrib     : tcGlobals -> Env.attrib_info -> attrib list -> int32 option

val tcref_bind_attrib        : tcGlobals -> Env.attrib_info -> tycon_ref -> ((Il.custom_attr_elem list * Il.custom_attr_named_arg list) -> 'a option) -> (attrib -> 'a option) -> 'a option
val tcref_has_attrib         : tcGlobals -> Env.attrib_info -> tycon_ref -> bool

val is_CompilationMappingAttr       : Il.custom_attr -> bool
val is_IntfDataVersionAttr          : Il.custom_attr -> bool
val is_matching_IntfDataVersionAttr : Il.version_info -> Il.custom_attr -> bool

val mk_CompilationMappingAttr  : tcGlobals -> int             -> Il.custom_attr
val mk_CompilationMappingAttrWithSeqNum  : tcGlobals -> int -> int             -> Il.custom_attr
val mk_CompilationMappingAttrWithVariantNumAndSeqNum  : tcGlobals -> int -> int -> int             -> Il.custom_attr
val mk_IntfDataVersionAttr     : tcGlobals -> Il.version_info -> Il.custom_attr
val mk_CompilerGeneratedAttr   : tcGlobals -> int             -> Il.custom_attr

(*-------------------------------------------------------------------------
 * More common type construction
 *------------------------------------------------------------------------- *)

val is_byref_ty : tcGlobals -> typ -> bool
val dest_byref_ty : tcGlobals -> typ -> typ

(*-------------------------------------------------------------------------
 * Tuple constructors/destructors
 *------------------------------------------------------------------------- *)

val typed_expr_for_val : range -> val_spec -> expr * typ

val is_tuple : expr -> bool
val try_dest_tuple : expr -> expr list
val mk_tupled : tcGlobals -> range -> expr list -> typ list -> expr 
val mk_tupled_ty : tcGlobals -> typ list -> typ
val mk_tupled_vars_ty : tcGlobals -> val_spec list -> typ
val mk_tupled_vars : tcGlobals -> range -> val_spec list -> expr 
val mk_meth_ty : tcGlobals -> typ list -> typ -> typ

(*-------------------------------------------------------------------------
 * 
 *------------------------------------------------------------------------- *)

val adjust_val_for_expected_arity : tcGlobals -> range -> val_ref -> val_use_flag -> val_arity -> expr * typ
val linearise_top_match : tcGlobals -> expr -> expr

(*-------------------------------------------------------------------------
 * XMLDoc signatures, used by both VS mode and XML-help emit
 *------------------------------------------------------------------------- *)

val argsEnc : tcGlobals -> typars -> typ list -> string
val xmlDocSigOfVal : tcGlobals -> string -> val_spec -> string
val xmlDocSigOfTycon : tcGlobals -> string -> tycon_spec -> string
val xmlDocSigOfSubModul : tcGlobals -> string -> string

(*---------------------------------------------------------------------------
 * From TAST tycon_ref to IL type_ref
 *------------------------------------------------------------------------- *)

val il_repr_of_tcref : tycon_ref -> il_type_repr

(*---------------------------------------------------------------------------
 * Resolve static optimizations
 *------------------------------------------------------------------------- *)

val static_optimization_constraint_definitely_satisfied : Env.tcGlobals -> Tast.tstatic_optimization_constraint -> bool 
val static_optimization_constraint_definitely_false     : Env.tcGlobals -> Tast.tstatic_optimization_constraint -> bool 
val mk_static_optimization_expr     : Env.tcGlobals -> tstatic_optimization_constraint list * expr * expr * range -> expr

(*---------------------------------------------------------------------------
 * Build for loops
 *------------------------------------------------------------------------- *)

val mk_fast_for_loop : Env.tcGlobals -> range -> val_spec -> expr -> bool -> expr -> expr -> expr

(*---------------------------------------------------------------------------
 * Active pattern helpers
 *------------------------------------------------------------------------- *)

val name_of_apref : apelem_ref -> string
val apinfo_of_vname : string -> apinfo option
val apinfo_of_vref  : val_ref -> apinfo option
val mk_choices_ucref : Env.tcGlobals -> int -> int -> Tast.unionconstr_ref
val mk_choices_typ   : Env.tcGlobals -> Tast.tinst -> Tast.typ

val names_of_apinfo      : apinfo -> string list 
val total_of_apinfo      : apinfo -> bool
val mk_apinfo_result_typ : Env.tcGlobals -> apinfo -> Tast.typ list-> Tast.typ
val mk_apinfo_typ        : Env.tcGlobals -> apinfo -> Tast.typ -> Tast.typ list-> Tast.typ

(*---------------------------------------------------------------------------
 * Structural rewrites
 *------------------------------------------------------------------------- *)

type rwEnv = {pre_intercept: ((expr -> expr) -> expr -> expr option) option;
              post_transform: expr -> expr option;
              underQuotations: bool }    

val rewrite_expr : rwEnv -> expr -> expr
val rewrite_assembly : rwEnv -> typedAssembly -> typedAssembly

val is_poly_constrained_val: val_spec -> bool
val tcaug_has_interface : Env.tcGlobals -> tycon_augmentation -> Tast.typ -> bool
val tcaug_has_override : Env.tcGlobals -> tycon_augmentation -> string -> Tast.typ list -> bool
