/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    https://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.api.ldap.model.schema.registries;


import java.util.Iterator;

import org.apache.directory.api.i18n.I18n;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.exception.LdapUnwillingToPerformException;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.api.ldap.model.schema.DitStructureRule;
import org.apache.directory.api.ldap.model.schema.SchemaObjectType;


/**
 * An immutable wrapper of the DitStructureRule registry.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class ImmutableDitStructureRuleRegistry implements DitStructureRuleRegistry
{
    /** The wrapped DitStructureRule registry */
    DitStructureRuleRegistry immutableDITStructureRuleRegistry;


    /**
     * Creates a new instance of ImmutableDitStructureRuleRegistry.
     *
     * @param ditStructureRuleRegistry The wrapped DitStructureRule registry
     */
    public ImmutableDitStructureRuleRegistry( DitStructureRuleRegistry ditStructureRuleRegistry )
    {
        immutableDITStructureRuleRegistry = ditStructureRuleRegistry;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains( int ruleId )
    {
        return immutableDITStructureRuleRegistry.contains( ruleId );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Iterator<DitStructureRule> iterator()
    {
        return immutableDITStructureRuleRegistry.iterator();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Iterator<Integer> ruleIdIterator()
    {
        return immutableDITStructureRuleRegistry.ruleIdIterator();
    }


    /**
     * {@inheritDoc}
     */
    @Override
public String getSchemaName( int ruleId ) throws LdapException
    {
        return immutableDITStructureRuleRegistry.getSchemaName( ruleId );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void register( DitStructureRule ditStructureRule ) throws LdapException
    {
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public DitStructureRule lookup( int ruleId ) throws LdapException
    {
        return immutableDITStructureRuleRegistry.lookup( ruleId );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void unregister( int ruleId ) throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void unregisterSchemaElements( String schemaName ) throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void renameSchema( String originalSchemaName, String newSchemaName ) throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public ImmutableDitStructureRuleRegistry copy()
    {
        return ( ImmutableDitStructureRuleRegistry ) immutableDITStructureRuleRegistry.copy();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public int size()
    {
        return immutableDITStructureRuleRegistry.size();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains( String oid )
    {
        return immutableDITStructureRuleRegistry.contains( oid );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getOidByName( String name ) throws LdapException
    {
        return immutableDITStructureRuleRegistry.getOidByName( name );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String getSchemaName( String oid ) throws LdapException
    {
        return immutableDITStructureRuleRegistry.getSchemaName( oid );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public SchemaObjectType getType()
    {
        return immutableDITStructureRuleRegistry.getType();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public DitStructureRule lookup( String oid ) throws LdapException
    {
        return immutableDITStructureRuleRegistry.lookup( oid );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public Iterator<String> oidsIterator()
    {
        return immutableDITStructureRuleRegistry.oidsIterator();
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public DitStructureRule unregister( String numericOid ) throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public DitStructureRule get( String oid )
    {
        return immutableDITStructureRuleRegistry.get( oid );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void clear() throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public DitStructureRule unregister( DitStructureRule schemaObject ) throws LdapException
    {
        throw new LdapUnwillingToPerformException( ResultCodeEnum.NO_SUCH_OPERATION, I18n.err( I18n.ERR_13704_CANNOT_MODIFY_DITSR_REGISTRY_COPY ) );
    }
}
