/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.plugins.pmd;

import java.io.File;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

/**
 * @author <a href="mailto:oching@apache.org">Maria Odea Ching</a>
 * @version $Id$
 */
public class PmdViolationCheckMojoTest extends AbstractPmdReportTestCase {

    public void testDefaultConfiguration() throws Exception {
        generateReport("pmd", "default-configuration/default-configuration-plugin-config.xml");

        try {
            final File testPom = new File(
                    getBasedir(),
                    "src/test/resources/unit/default-configuration/pmd-check-default-configuration-plugin-config.xml");
            final PmdViolationCheckMojo mojo = lookupMojo(getGoal(), testPom);
            mojo.execute();

            fail("MojoFailureException should be thrown.");
        } catch (final MojoFailureException e) {
            assertTrue(
                    e.getMessage().startsWith("PMD " + AbstractPmdReport.getPmdVersion() + " has found 8 violations."));
        }
    }

    public void testNotFailOnViolation() throws Exception {
        generateReport("pmd", "default-configuration/default-configuration-plugin-config.xml");

        File testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-notfailonviolation-plugin-config.xml");
        final PmdViolationCheckMojo pmdViolationMojo = lookupMojo(getGoal(), testPom);
        pmdViolationMojo.execute();
    }

    public void testMaxAllowedViolations() throws Exception {
        generateReport("pmd", "default-configuration/default-configuration-plugin-config.xml");

        File testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-notfailmaxviolation-plugin-config.xml");
        final PmdViolationCheckMojo pmdViolationMojo = lookupMojo(getGoal(), testPom);
        pmdViolationMojo.execute();

        testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-failmaxviolation-plugin-config.xml");
        final PmdViolationCheckMojo pmdViolationMojoFail = lookupMojo(getGoal(), testPom);

        try {
            pmdViolationMojoFail.execute();
            fail("Exception Expected");
        } catch (final MojoFailureException e) {
            assertTrue(e.getMessage()
                    .startsWith("PMD " + AbstractPmdReport.getPmdVersion()
                            + " has found 5 violations and issued 3 warnings."));
        }
    }

    public void testFailurePriority() throws Exception {
        generateReport("pmd", "default-configuration/default-configuration-plugin-config.xml");

        File testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-failonpriority-plugin-config.xml");
        PmdViolationCheckMojo pmdViolationCheckMojo = lookupMojo(getGoal(), testPom);
        pmdViolationCheckMojo.execute();

        testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-failandwarnonpriority-plugin-config.xml");
        pmdViolationCheckMojo = lookupMojo(getGoal(), testPom);

        try {
            pmdViolationCheckMojo.execute();
            fail("MojoFailureException expected");
        } catch (final MojoFailureException e) {
            assertTrue(e.getMessage()
                    .startsWith("PMD " + AbstractPmdReport.getPmdVersion()
                            + " has found 5 violations and issued 3 warnings."));
        }
    }

    public void testException() throws Exception {
        try {
            final File testPom = new File(
                    getBasedir(),
                    "src/test/resources/unit/custom-configuration/pmd-check-exception-test-plugin-config.xml");
            final PmdViolationCheckMojo mojo = lookupMojo(getGoal(), testPom);
            mojo.project = new MavenProject();
            mojo.execute();

            fail("MojoFailureException should be thrown.");
        } catch (final MojoFailureException e) {
            assertNotNull(e.getMessage());
        }
    }

    public void testViolationExclusion() throws Exception {
        generateReport("pmd", "default-configuration/default-configuration-plugin-config.xml");

        File testPom = new File(
                getBasedir(),
                "src/test/resources/unit/default-configuration/pmd-check-pmd-exclusions-configuration-plugin-config.xml");
        final PmdViolationCheckMojo pmdViolationMojo = lookupMojo(getGoal(), testPom);

        // this call shouldn't throw an exception, as the classes with violations have been excluded
        pmdViolationMojo.execute();
    }

    @Override
    protected String getGoal() {
        return "check";
    }
}
