/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.daemon.logviewer.utils;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.function.BinaryOperator;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.storm.StormTimer;
import org.apache.storm.daemon.logviewer.utils.DeletionMeta;
import org.apache.storm.daemon.logviewer.utils.DirectoryCleaner;
import org.apache.storm.daemon.logviewer.utils.WorkerLogs;
import org.apache.storm.metric.StormMetricsRegistry;
import org.apache.storm.utils.ObjectReader;
import org.apache.storm.utils.Time;
import org.apache.storm.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogCleaner
implements Runnable,
Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(LogCleaner.class);
    private final Timer cleanupRoutineDuration;
    private final Histogram numFilesCleanedUp;
    private final Histogram diskSpaceFreed;
    private final Meter numFileRemovalExceptions;
    private final Meter numCleanupExceptions;
    private final Map<String, Object> stormConf;
    private final Integer intervalSecs;
    private final Path logRootDir;
    private final DirectoryCleaner directoryCleaner;
    private final WorkerLogs workerLogs;
    private StormTimer logviewerCleanupTimer;
    private final long maxSumWorkerLogsSizeMb;
    private long maxPerWorkerLogsSizeMb;

    public LogCleaner(Map<String, Object> stormConf, WorkerLogs workerLogs, DirectoryCleaner directoryCleaner, Path logRootDir, StormMetricsRegistry metricsRegistry) {
        this.stormConf = stormConf;
        this.intervalSecs = ObjectReader.getInt((Object)stormConf.get("logviewer.cleanup.interval.secs"), null);
        this.logRootDir = logRootDir;
        this.workerLogs = workerLogs;
        this.directoryCleaner = directoryCleaner;
        this.maxSumWorkerLogsSizeMb = ObjectReader.getInt((Object)stormConf.get("logviewer.max.sum.worker.logs.size.mb")).intValue();
        this.maxPerWorkerLogsSizeMb = ObjectReader.getInt((Object)stormConf.get("logviewer.max.per.worker.logs.size.mb")).intValue();
        this.maxPerWorkerLogsSizeMb = Math.min(this.maxPerWorkerLogsSizeMb, (long)((double)this.maxSumWorkerLogsSizeMb * 0.5));
        LOG.info("configured max total size of worker logs: {} MB, max total size of worker logs per directory: {} MB", (Object)this.maxSumWorkerLogsSizeMb, (Object)this.maxPerWorkerLogsSizeMb);
        metricsRegistry.registerGauge("logviewer:worker-log-dir-size", () -> this.sizeOfDir(logRootDir));
        this.cleanupRoutineDuration = metricsRegistry.registerTimer("logviewer:cleanup-routine-duration-ms");
        this.numFilesCleanedUp = metricsRegistry.registerHistogram("logviewer:num-files-cleaned-up");
        this.diskSpaceFreed = metricsRegistry.registerHistogram("logviewer:disk-space-freed-in-bytes");
        this.numFileRemovalExceptions = metricsRegistry.registerMeter("logviewer:num-file-removal-exceptions");
        this.numCleanupExceptions = metricsRegistry.registerMeter("logviewer:num-other-cleanup-exceptions");
    }

    private long sizeOfDir(Path dir) {
        long l;
        block8: {
            Stream<Path> stream = Files.walk(dir, new FileVisitOption[0]);
            try {
                l = stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).mapToLong(p -> p.toFile().length()).sum();
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    LOG.debug("Failed to get size of directory {}", (Object)dir);
                    return 0L;
                }
            }
            stream.close();
        }
        return l;
    }

    public void start() {
        if (this.intervalSecs != null) {
            LOG.debug("starting log cleanup thread at interval: {}", (Object)this.intervalSecs);
            this.logviewerCleanupTimer = new StormTimer("logviewer-cleanup", (t, e) -> {
                LOG.error("Error when doing logs cleanup", e);
                Utils.exitProcess((int)20, (String)"Error when doing log cleanup");
            });
            this.logviewerCleanupTimer.scheduleRecurring(0, this.intervalSecs.intValue(), (Runnable)this);
        } else {
            LOG.warn("The interval for log cleanup is not set. Skip starting log cleanup thread.");
        }
    }

    @Override
    public void close() {
        if (this.logviewerCleanupTimer != null) {
            try {
                this.logviewerCleanupTimer.close();
            }
            catch (Exception ex) {
                throw Utils.wrapInRuntime((Exception)ex);
            }
        }
    }

    @Override
    public void run() {
        int numFilesCleaned = 0;
        long diskSpaceCleaned = 0L;
        try (Timer.Context t = this.cleanupRoutineDuration.time();){
            long nowMills = Time.currentTimeMillis();
            Set<Path> oldLogDirs = this.selectDirsForCleanup(nowMills);
            long nowSecs = TimeUnit.MILLISECONDS.toSeconds(nowMills);
            SortedSet<Path> deadWorkerDirs = this.getDeadWorkerDirs((int)nowSecs, oldLogDirs);
            LOG.debug("log cleanup: now={} old log dirs {} dead worker dirs {}", new Object[]{nowSecs, oldLogDirs.stream().map(p -> p.getFileName().toString()).collect(Collectors.joining(",")), deadWorkerDirs.stream().map(p -> p.getFileName().toString()).collect(Collectors.joining(","))});
            for (Path dir : deadWorkerDirs) {
                Path path = dir.toAbsolutePath().normalize();
                long sizeInBytes = this.sizeOfDir(dir);
                LOG.info("Cleaning up: Removing {}, {} KB", (Object)path, (Object)((double)sizeInBytes * 0.001));
                try {
                    Utils.forceDelete((String)path.toString());
                    this.cleanupEmptyTopoDirectory(dir);
                    ++numFilesCleaned;
                    diskSpaceCleaned += sizeInBytes;
                }
                catch (Exception ex) {
                    this.numFileRemovalExceptions.mark();
                    LOG.error(ex.getMessage(), (Throwable)ex);
                }
            }
            List<DeletionMeta> perWorkerDirCleanupMeta = this.perWorkerDirCleanup(this.maxPerWorkerLogsSizeMb * 1024L * 1024L);
            numFilesCleaned += perWorkerDirCleanupMeta.stream().mapToInt(meta -> meta.deletedFiles).sum();
            diskSpaceCleaned += perWorkerDirCleanupMeta.stream().mapToLong(meta -> meta.deletedSize).sum();
            DeletionMeta globalLogCleanupMeta = this.globalLogCleanup(this.maxSumWorkerLogsSizeMb * 1024L * 1024L);
            numFilesCleaned += globalLogCleanupMeta.deletedFiles;
            diskSpaceCleaned += globalLogCleanupMeta.deletedSize;
        }
        catch (Exception ex) {
            this.numCleanupExceptions.mark();
            LOG.error("Exception while cleaning up old log.", (Throwable)ex);
        }
        this.numFilesCleanedUp.update(numFilesCleaned);
        this.diskSpaceFreed.update(diskSpaceCleaned);
    }

    @VisibleForTesting
    List<DeletionMeta> perWorkerDirCleanup(long size) {
        return this.workerLogs.getAllWorkerDirs().stream().map(dir -> {
            try {
                return this.directoryCleaner.deleteOldestWhileTooLarge(Collections.singletonList(dir), size, true, null);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).collect(Collectors.toList());
    }

    @VisibleForTesting
    DeletionMeta globalLogCleanup(long size) throws Exception {
        ArrayList<Path> workerDirs = new ArrayList<Path>(this.workerLogs.getAllWorkerDirs());
        SortedSet<Path> aliveWorkerDirs = this.workerLogs.getAliveWorkerDirs();
        return this.directoryCleaner.deleteOldestWhileTooLarge(workerDirs, size, false, aliveWorkerDirs);
    }

    @VisibleForTesting
    void cleanupEmptyTopoDirectory(Path dir) throws IOException {
        Path topoDir = dir.getParent();
        try (Stream<Path> topoDirContent = Files.list(topoDir);){
            if (!topoDirContent.findAny().isPresent()) {
                Utils.forceDelete((String)topoDir.toAbsolutePath().normalize().toString());
            }
        }
    }

    @VisibleForTesting
    SortedSet<Path> getDeadWorkerDirs(int nowSecs, Set<Path> logDirs) throws Exception {
        if (logDirs.isEmpty()) {
            return new TreeSet<Path>();
        }
        Set<String> aliveIds = this.workerLogs.getAliveIds(nowSecs);
        return this.workerLogs.getLogDirs(logDirs, wid -> !aliveIds.contains(wid));
    }

    @VisibleForTesting
    Set<Path> selectDirsForCleanup(long nowMillis) {
        Set set;
        block8: {
            Predicate<Path> fileFilter = this.mkFileFilterForLogCleanup(nowMillis);
            Stream<Path> fileList = Files.list(this.logRootDir);
            try {
                set = fileList.flatMap(a -> {
                    try {
                        return Files.list(a);
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }).filter(fileFilter).collect(Collectors.toCollection(TreeSet::new));
                if (fileList == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (fileList != null) {
                        try {
                            fileList.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw Utils.wrapInRuntime((Exception)e);
                }
            }
            fileList.close();
        }
        return set;
    }

    @VisibleForTesting
    Predicate<Path> mkFileFilterForLogCleanup(long nowMillis) {
        return file -> Files.isDirectory(file, LinkOption.NOFOLLOW_LINKS) && this.lastModifiedTimeWorkerLogdir((Path)file) <= this.cleanupCutoffAgeMillis(nowMillis);
    }

    private long lastModifiedTimeWorkerLogdir(Path logDir) {
        long l;
        block11: {
            long dirModified = Files.getLastModifiedTime(logDir, new LinkOption[0]).toMillis();
            DirectoryStream<Path> dirStream = this.directoryCleaner.getStreamForDirectory(logDir);
            try {
                l = StreamSupport.stream(dirStream.spliterator(), false).map(p -> {
                    try {
                        return Files.getLastModifiedTime(p, new LinkOption[0]).toMillis();
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }).reduce(dirModified, BinaryOperator.maxBy(Long::compareTo));
                if (dirStream == null) break block11;
            }
            catch (Throwable throwable) {
                try {
                    try {
                        if (dirStream != null) {
                            try {
                                dirStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        LOG.error(e.getMessage(), (Throwable)e);
                        return dirModified;
                    }
                }
                catch (IOException e) {
                    throw Utils.wrapInRuntime((Exception)e);
                }
            }
            dirStream.close();
        }
        return l;
    }

    @VisibleForTesting
    long cleanupCutoffAgeMillis(long nowMillis) {
        Integer intervalMins = ObjectReader.getInt((Object)this.stormConf.get("logviewer.cleanup.age.mins"));
        return nowMillis - TimeUnit.MINUTES.toMillis(intervalMins.intValue());
    }
}

