/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: Control.java
 * Written by Steven M. Rubin
 *
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.tool.placement.general;

import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.hierarchy.Library;
import com.sun.electric.database.id.CellUsage;
import com.sun.electric.database.topology.NodeInst;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.placement.Placement;
import com.sun.electric.tool.placement.Placement.PlacementPreferences;
import com.sun.electric.tool.placement.PlacementAdapter;
import com.sun.electric.tool.placement.PlacementFrameElectric;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Implementation of the overall floorplanning and placement process, using multiple algorithms.
 */
public class Control extends PlacementFrameElectric
{
    @Override
	public String getAlgorithmName() { return "General Placement"; }
    
	/**
	 * Method to do placement by whatever method is appropriate.
	 * @param placementNodes a list of all nodes that are to be placed.
	 * @param allNetworks a list of all networks that connect the nodes.
	 * @param cellName the name of the cell being placed.
	 * @param job the Job (for testing abort).
	 */
    @Override
	public void runPlacement(List<PlacementNode> placementNodes, List<PlacementNetwork> allNetworks, String cellName, Job job)
	{
    	// distribute the editing preferences to the placement algorithms
		PlacementAdapter.BUpa.setEditingPreferences(ep);
		PlacementAdapter.BUpl.setEditingPreferences(ep);

		// check size of placement task
		if (placementNodes.size() > 100)
		{
			// large task: start by decomposing into smaller ones in a new library
			System.out.println("Large placement task: breaking it into smaller tasks...");
			Library lib = PlacementAdapter.BUpa.doBottomUp(placementNodes, allNetworks);
			System.out.println("Placement decomposed into new library: " + lib.getName());
			System.out.println();

			// now place the smaller cells in the decomposed library
			Map<Cell,Cell> placedCells = new HashMap<Cell,Cell>();
			List<Cell> regularCells = new ArrayList<Cell>();
			List<Cell> irregularCells = new ArrayList<Cell>();
			Cell allClusters = null;
			for(Iterator<Cell> it = lib.getCells(); it.hasNext(); )
			{
				Cell cell = it.next();
				if (cell.getName().equals("ALLCLUSTERS")) { allClusters = cell;   continue; }
				Boolean useColumns = getColumnPlacement(cell);
				if (useColumns == null) irregularCells.add(cell); else
					regularCells.add(cell);
			}

			// use Force-Directed-Row/Column on fixed-size placement tasks
			PlacementPreferences ppRowCol = new PlacementPreferences(false);
			ppRowCol.placementAlgorithm = PlacementAdapter.FD3.getAlgorithmName();
			PlacementAdapter.FD3.makeStacksEven.setValue(Boolean.TRUE);
			for(Cell cell : regularCells)
			{
				Cell newCell = Placement.placeCellNoJob(cell, ep, ppRowCol, true, job);
				if (newCell != null) placedCells.put(cell, newCell);
			}

			// use Bottom-Up Placement on irregular-size placement tasks
			PlacementPreferences ppIrregular = new PlacementPreferences(false);
			ppIrregular.placementAlgorithm = PlacementAdapter.BUpl.getAlgorithmName();
			for(Cell cell : irregularCells)
			{
				Cell newCell = Placement.placeCellNoJob(cell, ep, ppIrregular, true, job);
				if (newCell != null) placedCells.put(cell, newCell);
			}

			// now rebuild the "ALLCLUSTERS" cell
			if (allClusters != null)
			{
				for(Iterator<NodeInst> it = allClusters.getNodes(); it.hasNext(); )
				{
					NodeInst ni = it.next();
					if (!ni.isCellInstance()) continue;
					Cell np = (Cell)ni.getProto();
					Cell newCell = placedCells.get(np);
					if (newCell == null) continue;
					if (ni.replace(newCell, ep, false, false) == null)
						System.out.println("ERROR: Could not replace node " + ni.describe(false) + " in cell " +
							allClusters.describe(false) + " with cell " + newCell.describe(false));
				}

				// delete unplaced cells
				for(Cell cell : placedCells.keySet())
				{
			        Iterator<CellUsage> it = cell.getUsagesOf();
			        if (!it.hasNext()) cell.kill();
				}

				// now place the top-level cell (TODO: this step takes too long)
				setRedispCell(allClusters);
				PlacementPreferences ppToplevel = new PlacementPreferences(false);
				ppToplevel.placementAlgorithm = PlacementAdapter.BUpl.getAlgorithmName();
				Cell newCell = Placement.placeCellNoJob(allClusters, ep, ppToplevel, false, job);
				if (newCell != null) setRedispCell(newCell);
			}

			// set "failure" so that the original cell doesn't get rebuilt
			setFailure(true);
		} else
		{
			// cell is small: just do placement
			Boolean useColumns = RowCol.isColumnPlacement(placementNodes, null, null, true);
			if (useColumns == null)
			{
				// there are no common widths or heights: use Bottom-up placement
				System.out.println("General Placement algorithm chooses " + PlacementAdapter.BUpl.getAlgorithmName() +
					" to place the cell");
				PlacementAdapter.BUpl.runPlacement(placementNodes, allNetworks, cellName, job);
				return;	
			}

			// cell has fixed width/height nodes: use Force-Directed-Row/Column
			System.out.println("General Placement algorithm chooses " + PlacementAdapter.FD3.getAlgorithmName() +
				" to place the cell");
			PlacementAdapter.FD3.makeStacksEven.setValue(Boolean.TRUE);
			PlacementAdapter.FD3.runPlacement(placementNodes, allNetworks, cellName, job);
		}
	}

	/**
	 * Method to determine whether a Cell uses row or column placement.
	 * @param cell the Cell to be placed.
	 * @return TRUE to do column placement (all nodes are fixed width);
	 * FALSE to do row placement (all nodes are fixed height);
	 * null if nodes are mixed size.
	 */
	public Boolean getColumnPlacement(Cell cell)
	{
		// make sure cells have the same girth
		Double commonWid = new Double(-1), commonHei = new Double(-1);
		for(Iterator<NodeInst> it = cell.getNodes(); it.hasNext(); )
		{
			NodeInst ni = it.next();
			if (!ni.isCellInstance()) continue;
			if (commonWid != null)
			{
				if (commonWid.doubleValue() < 0) commonWid = new Double(ni.getXSize());
				if (commonWid.doubleValue() != ni.getXSize()) commonWid = null;
			}
			if (commonHei != null)
			{
				if (commonHei.doubleValue() < 0) commonHei = new Double(ni.getYSize());
				if (commonHei.doubleValue() != ni.getYSize()) commonHei = null;
			}
		}
		if (commonWid == null && commonHei == null)
		{
			// there are no common widths or heights
			return null;	
		}

		Boolean useColumns = Boolean.FALSE;
		if (commonWid != null && commonHei == null)
		{
			useColumns = Boolean.TRUE;
		}
		return useColumns;
	}
}
